#include "sys/win32/dark_theme.h"

#include <QApplication>
#include <QOperatingSystemVersion>
#include <QSettings>
#include <QWidget>

#include <iostream>
#include <memory>

#include <dwmapi.h>

struct LibraryDeconstructor {
		using pointer = HINSTANCE;
		void operator()(pointer t) const { ::FreeLibrary(t); };
};

using Library = std::unique_ptr<HINSTANCE, LibraryDeconstructor>;

class Dwmapi {
	public:
		Dwmapi() { library.reset( ::LoadLibraryW(L"dwmapi.dll")); }

		HRESULT SetWindowAttribute(HWND hWnd, DWORD key, LPCVOID data, DWORD sz_data) {
			if (!library.get())
				return E_POINTER;

			/* GCC throws a fit here because C/C++ lacks a "generic" function pointer type.
			   Ignore. */
			auto set_wind_attrib = reinterpret_cast<decltype(::DwmSetWindowAttribute)*>(GetProcAddress(library.get(), "DwmSetWindowAttribute"));
			if (!set_wind_attrib)
				return E_POINTER;

			return set_wind_attrib(hWnd, key, data, sz_data);
		}

	protected:
		Library library = nullptr;
};

Dwmapi dwmapi;

namespace win32 {

/* NOTE: explicit conversion from builtin `bool` and win32 `BOOL` IS allowed. */
bool SetTitleBarToBlack(QWidget* win, bool enabled) {
	BOOL b = enabled;

	/* MAGIC NUMBERS: 19 and 20 are both DWMWA_USE_IMMERSIVE_DARK_MODE.
	   For clarification: it's 20 on newer versions of windows (i.e. win11 and late win10),
	   but it's 19 on very old versions of win10 nobody ought to be using anymore. */
	{
		HRESULT result = dwmapi.SetWindowAttribute(reinterpret_cast<HWND>(win->winId()), 20, &b, sizeof(b));
		if (result == S_OK)
			return b;
	}

	{
		HRESULT result = dwmapi.SetWindowAttribute(reinterpret_cast<HWND>(win->winId()), 19, &b, sizeof(b));
		if (result == S_OK)
			return b;
	}

	return b;
}

void SetTitleBarsToBlack(bool enabled) {
	for (QWidget* widget : qApp->topLevelWidgets()) {
		SetTitleBarToBlack(widget, enabled);
	}
}

bool DarkThemeAvailable() {
	const auto& ver = QOperatingSystemVersion::current();
	return (ver.majorVersion() > 10) ? true : (ver.majorVersion() == 10 && ver.microVersion() >= 17763);
}

bool IsInDarkTheme() {
	QSettings settings("HKEY_CURRENT_USER\\Software\\Microsoft\\Windows\\CurrentVersion\\Themes\\Personalize",
	                   QSettings::NativeFormat);
	return settings.value("AppsUseLightTheme", 1).toInt() == 0;
}

} // namespace win32
