#include "gui/window.h"
#include "core/config.h"
#include "core/session.h"
#include "gui/dark_theme.h"
#include "gui/dialog/settings.h"
#include "gui/dialog/about.h"
#include "gui/pages/anime_list.h"
#include "gui/pages/now_playing.h"
#include "gui/pages/statistics.h"
#include "gui/widgets/sidebar.h"
#include "services/services.h"
#include <QApplication>
#include <QFile>
#include <QMainWindow>
#include <QMenuBar>
#include <QPlainTextEdit>
#include <QStackedWidget>
#include <QTextStream>
#if MACOSX
#	include "sys/osx/dark_theme.h"
#elif defined(WIN32)
#	include "sys/win32/dark_theme.h"
#endif

enum class Pages {
	NOW_PLAYING,

	ANIME_LIST,
	HISTORY,
	STATISTICS,

	SEARCH,
	SEASONS,
	TORRENTS
};

MainWindow::MainWindow(QWidget* parent) : QMainWindow(parent) {
	main_widget = new QWidget(parent);

	SideBar* sidebar = new SideBar(main_widget);
	sidebar->AddItem(tr("Now Playing"), SideBar::CreateIcon(":/icons/16x16/film.png"));
	sidebar->AddSeparator();
	sidebar->AddItem(tr("Anime List"), SideBar::CreateIcon(":/icons/16x16/document-list.png"));
	sidebar->AddItem(tr("History"), SideBar::CreateIcon(":/icons/16x16/clock-history-frame.png"));
	sidebar->AddItem(tr("Statistics"), SideBar::CreateIcon(":/icons/16x16/chart.png"));
	sidebar->AddSeparator();
	sidebar->AddItem(tr("Search"), SideBar::CreateIcon(":/icons/16x16/magnifier.png"));
	sidebar->AddItem(tr("Seasons"), SideBar::CreateIcon(":/icons/16x16/calendar.png"));
	sidebar->AddItem(tr("Torrents"), SideBar::CreateIcon(":/icons/16x16/feed.png"));
	sidebar->setFixedWidth(128);
	sidebar->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding);

	QStackedWidget* stack = new QStackedWidget(main_widget);
	stack->addWidget(new NowPlayingWidget(main_widget));
	stack->addWidget(new AnimeListWidget(main_widget));
	stack->addWidget(new StatisticsWidget(main_widget));

	connect(sidebar, &SideBar::CurrentItemChanged, stack, [stack](int index) {
		switch (index) {
			case 0:
			case 1: stack->setCurrentIndex(index); break;
			case 3: stack->setCurrentIndex(2); break;
			default: break;
		}
	});
	sidebar->setCurrentRow(2);

	/* Menu Bar */
	QAction* action;
	QMenuBar* menubar = new QMenuBar(parent);
	QMenu* menu = menubar->addMenu(tr("&File"));

	QMenu* submenu = menu->addMenu(tr("&Library folders"));
	action = submenu->addAction(tr("&Add new folder..."));

	action = menu->addAction(tr("&Scan available episodes"));

	menu->addSeparator();

	action = menu->addAction(tr("Play &next episode"));
	action->setShortcut(QKeySequence(Qt::CTRL + Qt::Key_N));
	action = menu->addAction(tr("Play &random episode"));
	action->setShortcut(QKeySequence(Qt::CTRL + Qt::Key_R));

	menu->addSeparator();

	action = menu->addAction(tr("E&xit"), qApp, &QApplication::quit);

	menu = menubar->addMenu(tr("&Services"));
	action = menu->addAction(tr("Synchronize &list"), [this, stack] {
		Services::Synchronize();
		((AnimeListWidget*)stack->widget((int)Pages::ANIME_LIST))->Refresh();
	});
	action->setShortcut(QKeySequence(Qt::CTRL + Qt::Key_S));

	menu->addSeparator();

	submenu = menu->addMenu(tr("&AniList"));
	action = submenu->addAction(tr("Go to my &profile"));
	action = submenu->addAction(tr("Go to my &stats"));

	submenu = menu->addMenu(tr("&Kitsu"));
	action = submenu->addAction(tr("Go to my &feed"));
	action = submenu->addAction(tr("Go to my &library"));
	action = submenu->addAction(tr("Go to my &profile"));

	submenu = menu->addMenu(tr("&MyAnimeList"));
	action = submenu->addAction(tr("Go to my p&anel"));
	action = submenu->addAction(tr("Go to my &profile"));
	action = submenu->addAction(tr("Go to my &history"));

	menu = menubar->addMenu(tr("&Tools"));
	submenu = menu->addMenu(tr("&Export anime list"));
	action = submenu->addAction(tr("Export as &Markdown..."));
	action = submenu->addAction(tr("Export as MyAnimeList &XML..."));

	menu->addSeparator();

	action = menu->addAction(tr("Enable anime &recognition"));
	action->setCheckable(true);
	action = menu->addAction(tr("Enable auto &sharing"));
	action->setCheckable(true);
	action = menu->addAction(tr("Enable &auto synchronization"));
	action->setCheckable(true);

	menu->addSeparator();

	action = menu->addAction(tr("&Settings"), [this] {
		SettingsDialog dialog(this);
		dialog.exec();
	});
	action->setMenuRole(QAction::PreferencesRole);

	menu = menubar->addMenu(tr("&View"));

	std::map<QAction*, int> page_to_index_map = {};

	QActionGroup* pages_group = new QActionGroup(this);
	pages_group->setExclusive(true);

	action = pages_group->addAction(menu->addAction(tr("&Now Playing")));
	action->setCheckable(true);
	page_to_index_map[action] = 0;

	action = pages_group->addAction(menu->addAction(tr("&Anime List")));
	page_to_index_map[action] = 1;

	action->setCheckable(true);
	action->setChecked(true);
	action = pages_group->addAction(menu->addAction(tr("&History")));
	action->setCheckable(true);
	page_to_index_map[action] = 2;

	action = pages_group->addAction(menu->addAction(tr("&Statistics")));
	action->setCheckable(true);
	page_to_index_map[action] = 3;

	action = pages_group->addAction(menu->addAction(tr("S&earch")));
	action->setCheckable(true);
	page_to_index_map[action] = 4;

	action = pages_group->addAction(menu->addAction(tr("Se&asons")));
	action->setCheckable(true);
	page_to_index_map[action] = 5;

	action = pages_group->addAction(menu->addAction(tr("&Torrents")));
	action->setCheckable(true);
	page_to_index_map[action] = 6;

	connect(pages_group, &QActionGroup::triggered, this, [page_to_index_map, stack](QAction* action) {
		int index = page_to_index_map.at(action);
		switch (index) {
			case 0:
			case 1: stack->setCurrentIndex(index); break;
			case 3: stack->setCurrentIndex(2);
			default: break;
		}
	});
	menu->addSeparator();
	menu->addAction(tr("Show sidebar"));

	menu = menubar->addMenu(tr("&Help"));
	action = menu->addAction(tr("About Minori"), this, [this] {
		AboutWindow dialog(this);
		dialog.exec();
	});
	action = menu->addAction(tr("About &Qt"), qApp, &QApplication::aboutQt);
	action->setMenuRole(QAction::AboutQtRole);

	setMenuBar(menubar);

	QHBoxLayout* layout = new QHBoxLayout(main_widget);
	layout->addWidget(sidebar);
	layout->addWidget(stack);
	setCentralWidget(main_widget);

	DarkTheme::SetTheme(session.config.theme);
}

void MainWindow::SetActivePage(QWidget* page) {
	this->setCentralWidget(page);
}

void MainWindow::closeEvent(QCloseEvent* event) {
	session.config.Save();
	event->accept();
}

#include "gui/moc_window.cpp"
