#include "gui/widgets/anime_button.h"

#include "core/anime_db.h"
#include "core/session.h"
#include "core/strings.h"
#include "gui/widgets/elided_label.h"
#include "gui/widgets/poster.h"
#include "gui/widgets/text.h"

#include <QDate>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QWidget>

/* This widget is only used on the Seasons page. */

/***********************************\
*|---------| Title                  *
*|         |                        *
*|         | Aired                  *
*|         | Episodes               *
*| Poster  | Producers              *
*|         | Score                  *
*|         | Popularity             *
*|         |                        *
*|_________| Synopsis               *
\***********************************/

AnimeButton::AnimeButton(QWidget* parent) : QFrame(parent) {
	setFrameShadow(QFrame::Plain);
	setFrameShape(QFrame::Box);
	QHBoxLayout* ly = new QHBoxLayout(this);

	_poster = new Poster(this);
	_poster->setFixedSize(120, 170);
	_poster->SetClickable(false);
	ly->addWidget(_poster, 0, Qt::AlignTop);

	{
		QWidget* misc_section = new QWidget(this);
		misc_section->setFixedSize(354, 180);

		QVBoxLayout* misc_layout = new QVBoxLayout(misc_section);
		misc_layout->setContentsMargins(0, 0, 0, 0);

		_title = new QLabel("", misc_section);
		_title->setAutoFillBackground(true);
		_title->setContentsMargins(4, 4, 4, 4);
		_title->setStyleSheet("background-color: rgba(0, 245, 25, 50);");
		{
			QFont fnt(_title->font());
			fnt.setWeight(QFont::Bold);
			_title->setFont(fnt);
		}
		misc_layout->addWidget(_title);

		_info = new TextWidgets::LabelledParagraph(tr("Aired:\nEpisodes:\nGenres:\nProducers:\nScore:\nPopularity:"),
		                                           "\n\n\n\n\n", misc_section);
		{
			QFont fnt(_info->GetLabels()->font());
			fnt.setWeight(QFont::Bold);
			_info->GetLabels()->setFont(fnt);
		}
		_info->setContentsMargins(4, 0, 4, 0);
		_info->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Minimum);
		misc_layout->addWidget(_info);

		QWidget* dummy = new QWidget(misc_section);
		dummy->setContentsMargins(4, 0, 4, 0);
		QVBoxLayout* dummy_layout = new QVBoxLayout(dummy);
		dummy_layout->setSpacing(0);
		// dummy_layout->setContentsMargins(0, 0, 0, 0);
		dummy_layout->setContentsMargins(0, 0, 0, 0);

		_synopsis = new ElidedLabel("", dummy);
		_synopsis->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding);
		dummy_layout->addWidget(_synopsis);
		misc_layout->addWidget(dummy);

		ly->addWidget(misc_section, 0, Qt::AlignTop);
	}
}

AnimeButton::AnimeButton(const Anime::Anime& anime, QWidget* parent) : AnimeButton(parent) {
	SetAnime(anime);
}

void AnimeButton::SetAnime(const Anime::Anime& anime) {
	_poster->SetAnime(anime);
	_title->setText(Strings::ToQString(anime.GetUserPreferredTitle()));

	{
		const QLocale& locale = session.config.locale.GetLocale();
		_info->GetParagraph()->SetText(locale.toString(anime.GetAirDate().GetAsQDate(), "dd MMM yyyy") + "\n" +
		                               QString::number(anime.GetEpisodes()) + "\n" +
		                               Strings::ToQString(Strings::Implode(anime.GetGenres(), ", ")) + "\n" + "...\n" +
		                               QString::number(anime.GetAudienceScore()) + "%\n" + "...");
	}

	{
		QString synopsis = Strings::ToQString(anime.GetSynopsis());
		QFontMetrics metrics(_synopsis->font());
		_synopsis->SetText(Strings::ToQString(anime.GetSynopsis()));
	}
}