#include <algorithm>
#include <fcntl.h>
#include <filesystem>
#include <fstream>
#include <iostream>
#include <sstream>
#include <string>
#include <sys/stat.h>
#include <unistd.h>
#include <unordered_map>
#include <vector>
#include <cstring>
#include <dirent.h>

#include "animia/util.h"

#define PROC_LOCATION "/proc"

namespace animia::internal::linux {

/* this uses dirent instead of std::filesystem; it would make a bit
   more sense to use the latter, but this is platform dependent already :) */
std::vector<std::string> GetAllFilesInDir(const std::string& _dir) {
	std::vector<std::string> ret;

	DIR* dir = opendir(_dir.c_str());
	if (!dir)
		return ret;

	struct dirent* dp;
	while ((dp = readdir(dir)) != NULL) {
		if (!(!strcmp(dp->d_name, ".") || !strcmp(dp->d_name, "..")))
			ret.push_back(_dir + "/" + dp->d_name);
	}

	closedir(dir);
	return ret;
}

std::string Basename(const std::string& path) {
	return path.substr(path.find_last_of("/") + 1, path.length());
}

static bool IsRegularFile(std::string link) {
	struct stat sb;
	if (stat(link.c_str(), &sb) == -1)
		return false;
	return S_ISREG(sb.st_mode);
}

static bool AreFlagsOk(pid_t pid, int fd) {
	const std::string path = PROC_LOCATION "/" + std::to_string(pid) + "/fdinfo/" + std::to_string(fd);
	std::stringstream buffer(util::ReadFile(path));

	int flags = 0;
	for (std::string line; std::getline(buffer, line);) {
		/* FIXME: exception handling here!! */
		if (line.rfind("flags:", 0) == 0) {
			flags = std::stoi(line.substr(line.find_last_not_of("0123456789") + 1));
		}
	}
	if (flags & O_WRONLY || flags & O_RDWR)
		return false;
	return true;
}

static std::string GetFilenameFromFd(std::string link) {
	/* gets around stupid linux limitation where /proc doesn't
	   give actual filesize readings */
	size_t  exe_size = 1024;
	ssize_t exe_used;
	std::string ret;

	while (1) {
		ret = std::string(exe_size, '\0');
		exe_used = readlink(link.c_str(), &ret.front(), ret.length());
		if (exe_used == (ssize_t)-1)
			return NULL;

		if (exe_used < (ssize_t)1) {
			errno = ENOENT;
			return NULL;
		}

		if (exe_used < (ssize_t)(exe_size - 1))
			break;

		exe_size += 1024;
	}

	return ret.c_str();
}

bool GetAllPids(std::set<pid_t>& pids) {
	for (const auto& dir : get_all_files_in_dir(PROC_LOCATION)) {
		pid_t pid;
		try {
			pid = std::stoul(basename(dir));
		} catch (std::invalid_argument) {
			continue;
		}
		pids.push_back(pid);
	}
}

bool GetProcessName(pid_t pid, std::string& result) {
	const std::string path = PROC_LOCATION "/" + std::to_string(pid) + "/comm";

	std::string result = util::ReadFile(path);
	result.erase(std::remove(result.begin(), result.end(), '\n'), result.end());
}

bool EnumerateOpenFiles(const std::set<pid_t>& pids, std::vector<std::tuple<pid_t, std::string>>& files) {
	for (const auto& pid : pids) {
		const std::string path = PROC_LOCATION "/" + std::to_string(pid) + "/fd";

		for (const auto& dir : GetAllFilesInDir(path)) {
			if (!AreFlagsOk(pid, std::stoi(basename(dir))))
				continue;

			std::string name = GetFilenameFromFd(dir);

			if (!IsRegularFile(name))
				continue;

			files.push_back({pid, name});
		}
	}
	return true;
}

} // namespace animia::internal::linux
