#include "core/session.h"
#include "core/strings.h"
#include "gui/dialog/settings.h"
#include "track/types.h"
#include <QListWidget>
#include <QListWidgetItem>
#include <QGroupBox>
#include <QCheckBox>
#include <QLabel>
#include <QSizePolicy>
#include <QVBoxLayout>
#include <algorithm>

Q_DECLARE_METATYPE(Track::Types::MediaPlayer);

QWidget* SettingsPageRecognition::CreatePlayersWidget() {
	QWidget* result = new QWidget(this);
	result->setAutoFillBackground(true);
	result->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Maximum);

	QVBoxLayout* full_layout = new QVBoxLayout(result);

	{
		/* Feed link */
		QWidget* widget = new QWidget(result);
		QVBoxLayout* widget_layout = new QVBoxLayout(widget);

		QCheckBox* checkbox = new QCheckBox(result);
		checkbox->setText(tr("Enable media player detection"));
		checkbox->setCheckState(detect_media_players ? Qt::Checked : Qt::Unchecked);
		widget_layout->addWidget(checkbox);

		{
			QLabel* label = new QLabel(tr("Allowed media players:"), widget);
			widget_layout->addWidget(label);
		}

		{
			QListWidget* listwidget = new QListWidget(widget);
			for (size_t i = 0; i < players.size(); i++) {
				const auto& player = players[i];
				{
					QListWidgetItem* item = new QListWidgetItem(listwidget);
					item->setCheckState(player.GetEnabled() ? Qt::Checked : Qt::Unchecked);
					item->setText(Strings::ToQString(player.GetName() + " (" + player.GetExecutable() + ")"));
					item->setData(Qt::UserRole, QVariant::fromValue(i));
				}
			}
			connect(listwidget, &QListWidget::itemChanged, this, [this](QListWidgetItem* item){
				if (!item)
					return;
				size_t i = item->data(Qt::UserRole).toUInt();
				players[i].SetEnabled(item->checkState());
			});
			/* this is down here because the listwidget state depends on it */
			connect(checkbox, &QCheckBox::stateChanged, this, [this, listwidget](int state) {
				detect_media_players = (state == Qt::Checked);
				listwidget->setEnabled(detect_media_players);
			});
			listwidget->setEnabled(checkbox->checkState() == Qt::Checked);

			widget_layout->addWidget(listwidget);
		}

		full_layout->addWidget(widget);
	}

	{
		/* Feed link */
		QWidget* widget = new QWidget(result);
		QVBoxLayout* widget_layout = new QVBoxLayout(widget);

		{
			QLabel* label = new QLabel(tr("Allowed file extensions:"), widget);
			widget_layout->addWidget(label);
		}

		{
			QListWidget* listwidget = new QListWidget(widget);
			for (size_t i = 0; i < extensions.size(); i++) {
				const auto& extension = extensions[i];
				{
					QListWidgetItem* item = new QListWidgetItem(listwidget);
					item->setCheckState(extension.GetEnabled() ? Qt::Checked : Qt::Unchecked);
					item->setText(Strings::ToQString("." + extension.GetExtension()));
					item->setData(Qt::UserRole, QVariant::fromValue(i));
				}
			}
			connect(listwidget, &QListWidget::itemChanged, this, [this](QListWidgetItem* item){
				if (!item)
					return;
				size_t i = item->data(Qt::UserRole).toUInt();
				extensions[i].SetEnabled(item->checkState());
			});

			widget_layout->addWidget(listwidget);
		}

		full_layout->addWidget(widget);
	}

	full_layout->setSpacing(10);

	return result;
}

void SettingsPageRecognition::SaveInfo() {
	session.config.recognition.detect_media_players = detect_media_players;
	session.recognition.players = players;
	session.recognition.extensions = extensions;
}

SettingsPageRecognition::SettingsPageRecognition(QWidget* parent)
	: SettingsPage(parent, tr("Recognition")),
	  players(session.recognition.players),
	  extensions(session.recognition.extensions) {
	detect_media_players = session.config.recognition.detect_media_players;
	AddTab(CreatePlayersWidget(), tr("Media players"));
}
