#ifndef __core__strings_h
#define __core__strings_h

#include <string>
#include <vector>
#include <array>

class QString;

namespace Strings {

template<unsigned...>struct seq{using type=seq;};
template<unsigned N, unsigned... Is>
struct gen_seq_x : gen_seq_x<N-1, N-1, Is...>{};
template<unsigned... Is>
struct gen_seq_x<0, Is...> : seq<Is...>{};
template<unsigned N>
using gen_seq=typename gen_seq_x<N>::type;

template<size_t S>
using size=std::integral_constant<size_t, S>;

template<class T, size_t N>
constexpr size<N> length( T const(&)[N] ) { return {}; }
template<class T, size_t N>
constexpr size<N> length( std::array<T, N> const& ) { return {}; }

template<class T>
using length_t = decltype(length(std::declval<T>()));

constexpr size_t string_size() { return 0; }
template<class...Ts>
constexpr size_t string_size( size_t i, Ts... ts ) {
  return (i?i-1:0) + string_size(ts...);
}
template<class...Ts>
using string_length=size< string_size( length_t<Ts>{}... )>;

template<class...Ts>
using combined_string = std::array<char, string_length<Ts...>{}+1>;

template<class Lhs, class Rhs, unsigned...I1, unsigned...I2>
constexpr const combined_string<Lhs,Rhs>
concat_impl( Lhs const& lhs, Rhs const& rhs, seq<I1...>, seq<I2...>)
{
    return {{ lhs[I1]..., rhs[I2]..., '\0' }};
}

template<class Lhs, class Rhs>
constexpr const combined_string<Lhs,Rhs>
concat(Lhs const& lhs, Rhs const& rhs)
{
    return concat_impl(lhs, rhs, gen_seq<string_length<Lhs>{}>{}, gen_seq<string_length<Rhs>{}>{});
}

template<class T0, class T1, class... Ts>
constexpr const combined_string<T0, T1, Ts...>
concat(T0 const&t0, T1 const&t1, Ts const&...ts)
{
    return concat(t0, concat(t1, ts...));
}

template<class T>
constexpr const combined_string<T>
concat(T const&t) {
    return concat(t, "");
}
constexpr const combined_string<>
concat() {
    return concat("");
}

/* Implode function: takes a vector of strings and turns it
   into a string, separated by delimiters. */
std::string Implode(const std::vector<std::string>& vector, const std::string& delimiter);

/* Substring removal functions */
std::string ReplaceAll(std::string string, const std::string& find, const std::string& replace);
std::string SanitizeLineEndings(const std::string& string);
std::string RemoveHtmlTags(std::string string);
std::string ParseHtmlEntities(std::string string);

/* stupid HTML bullshit */
std::string TextifySynopsis(const std::string& string);

std::string ToUpper(const std::string& string);
std::string ToLower(const std::string& string);

/* functions that make the way we convert from and to
   different string formats universal */
std::wstring ToWstring(const std::string& string);
std::wstring ToWstring(const QString& string);
std::string ToUtf8String(const std::wstring& wstring);
std::string ToUtf8String(const QString& string);
QString ToQString(const std::string& string);
QString ToQString(const std::wstring& wstring);

/* arithmetic :) */
int ToInt(const std::string& str, int def = 0);

bool BeginningMatchesSubstring(const std::string& str, const std::string& sub);

}; // namespace Strings

#endif // __core__strings_h