#include "core/http.h"
#include "core/session.h"
#include <QByteArray>
#include <QMessageBox>
#include <QThread>
#include <curl/curl.h>
#include <string>
#include <vector>

namespace HTTP {

HttpGetThread::HttpGetThread(std::string url, std::vector<std::string> headers, QObject* parent) : QThread(parent) {
	_url = url;
	_headers = headers;
}

size_t HttpGetThread::WriteCallback(void* contents, size_t size, size_t nmemb, void* userdata) {
	reinterpret_cast<QByteArray*>(userdata)->append(reinterpret_cast<char*>(contents), size * nmemb);
	return size * nmemb;
}

void HttpGetThread::run() {
	struct curl_slist* list = NULL;
	QByteArray userdata;

	CURL* curl = curl_easy_init();
	if (curl) {
		for (const std::string& h : _headers) {
			list = curl_slist_append(list, h.c_str());
		}
		curl_easy_setopt(curl, CURLOPT_URL, _url.c_str());
		curl_easy_setopt(curl, CURLOPT_WRITEDATA, &userdata);
		curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, &WriteCallback);
		/* Use system certs... useful on Windows. */
		curl_easy_setopt(curl, CURLOPT_SSL_OPTIONS, CURLSSLOPT_NATIVE_CA);
		CURLcode res = curl_easy_perform(curl);
		session.IncrementRequests();
		curl_easy_cleanup(curl);
		if (res != CURLE_OK) {
			QMessageBox box(QMessageBox::Icon::Critical, "",
			                QString("curl_easy_perform(curl) failed!: ") + QString(curl_easy_strerror(res)));
			box.exec();
		}
	}
	emit resultReady(userdata);
}

static size_t CurlWriteStringCallback(void* contents, size_t size, size_t nmemb, void* userdata) {
	reinterpret_cast<std::string*>(userdata)->append(reinterpret_cast<char*>(contents), size * nmemb);
	return size * nmemb;
}

/* Performs a basic (blocking) post request */
std::string PerformBasicPostRequest(std::string url, std::string data, std::vector<std::string> headers) {
	struct curl_slist* list = NULL;
	std::string userdata;
	CURL* curl = curl_easy_init();
	if (curl) {
		for (const std::string& h : headers) {
			list = curl_slist_append(list, h.c_str());
		}
		curl_easy_setopt(curl, CURLOPT_URL, url.c_str());
		curl_easy_setopt(curl, CURLOPT_POSTFIELDS, data.c_str());
		curl_easy_setopt(curl, CURLOPT_HTTPHEADER, list);
		curl_easy_setopt(curl, CURLOPT_WRITEDATA, &userdata);
		curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, &CurlWriteStringCallback);
		/* Use system certs... useful on Windows. */
		curl_easy_setopt(curl, CURLOPT_SSL_OPTIONS, CURLSSLOPT_NATIVE_CA);
		CURLcode res = curl_easy_perform(curl);
		session.IncrementRequests();
		curl_slist_free_all(list);
		curl_easy_cleanup(curl);
		if (res != CURLE_OK) {
			QMessageBox box(QMessageBox::Icon::Critical, "",
			                QString("curl_easy_perform(curl) failed!: ") + QString(curl_easy_strerror(res)));
			box.exec();
			return "";
		}
		return userdata;
	}
	return "";
}

} // namespace HTTP

#include "core/moc_http.cpp"
