#include "core/session.h"
#include "core/strings.h"
#include "gui/dialog/settings.h"
#include "gui/widgets/drop_list_widget.h"

#include <QCheckBox>
#include <QDir>
#include <QFileDialog>
#include <QGroupBox>
#include <QLabel>
#include <QListWidget>
#include <QListWidgetItem>
#include <QPushButton>
#include <QSizePolicy>
#include <QVBoxLayout>

QWidget* SettingsPageLibrary::CreateFoldersWidget() {
	QWidget* result = new QWidget(this);
	result->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Maximum);

	QVBoxLayout* full_layout = new QVBoxLayout(result);

	{
		QGroupBox* group_box = new QGroupBox(tr("Library folders"), result);
		group_box->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Maximum);

		QVBoxLayout* group_box_layout = new QVBoxLayout(group_box);

		{
			QLabel* label = new QLabel(tr("These folders will be scanned and monitored for new episodes."), group_box);
			group_box_layout->addWidget(label);
		}

		{
			DroppableListWidget* listwidget = new DroppableListWidget(group_box);
			listwidget->setSelectionMode(QAbstractItemView::ExtendedSelection);

			for (const auto& path : paths) {
				QListWidgetItem* item = new QListWidgetItem(listwidget);
				item->setText(Strings::ToQString(path));
				/* add icons as well soon */
			}

			connect(listwidget, &DroppableListWidget::FilesDropped, this, [this, listwidget](QStringList list) {
				for (const auto& dir : list) {
					paths.insert(Strings::ToUtf8String(dir));
					QListWidgetItem* item = new QListWidgetItem(listwidget);
					item->setText(dir);
				}
			});

			group_box_layout->addWidget(listwidget);

			{
				QWidget* widget = new QWidget(group_box);
				QHBoxLayout* widget_layout = new QHBoxLayout(widget);

				{
					QLabel* label = new QLabel(tr("Tip: You can drag and drop folders here."), widget);
					widget_layout->addWidget(label);
				}

				widget_layout->addStretch();

				{
					QPushButton* button = new QPushButton(tr("Add new..."), widget);

					connect(button, &QPushButton::clicked, this, [this, listwidget] {
						const QString dir = QFileDialog::getExistingDirectory(
						    this, tr("Open Directory"), QDir::homePath(),
						    QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks);
						const std::string s_dir = Strings::ToUtf8String(dir);
						if (dir.isEmpty() || paths.count(s_dir))
							return;

						paths.insert(s_dir);
						QListWidgetItem* item = new QListWidgetItem(listwidget);
						item->setText(dir);
					});

					widget_layout->addWidget(button);
				}

				{
					QPushButton* button = new QPushButton(tr("Remove"), widget);

					connect(listwidget, &QListWidget::itemSelectionChanged, this, [button, listwidget] {
						QList<QListWidgetItem*> selection = listwidget->selectedItems();
						button->setEnabled(selection.size() > 0);
					});

					connect(button, &QPushButton::clicked, this, [this, listwidget] {
						QList<QListWidgetItem*> selection = listwidget->selectedItems();
						for (const auto& item : selection) {
							paths.erase(Strings::ToUtf8String(item->text()));
							delete item;
						}
					});

					widget_layout->addWidget(button);
				}

				group_box_layout->addWidget(widget);
			}
		}

		full_layout->addWidget(group_box);
	}

	{
		QGroupBox* group_box = new QGroupBox(tr("Real-time monitor"), result);
		group_box->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Maximum);

		QVBoxLayout* group_box_layout = new QVBoxLayout(group_box);

		{
			QCheckBox* checkbox = new QCheckBox(tr("Detect new files and folders under library folders"), group_box);
			checkbox->setCheckState(real_time_monitor ? Qt::Checked : Qt::Unchecked);

			connect(checkbox, &QCheckBox::stateChanged, this,
			        [this](int state) { real_time_monitor = (state != Qt::Unchecked); });

			group_box_layout->addWidget(checkbox);
		}

		full_layout->addWidget(group_box);
	}

	full_layout->setSpacing(10);
	full_layout->addStretch();

	return result;
}

void SettingsPageLibrary::SaveInfo() {
	session.config.library.paths = paths;
	session.config.library.real_time_monitor = real_time_monitor;
}

SettingsPageLibrary::SettingsPageLibrary(QWidget* parent)
    : SettingsPage(parent, tr("Library")), paths(session.config.library.paths) {
	real_time_monitor = session.config.library.real_time_monitor;
	AddTab(CreateFoldersWidget(), tr("Folder"));
}
