/**
 * bsd.cpp 
 *  - provides support for most* versions of BSD
 *  - this *should* also work for OS X
 * more technical details: this is essentially a wrapper
 * around the very C-like BSD system functions that are...
 * kind of unnatural to use in modern C++.
**/
#include <vector>
#include <string>
#include <unordered_map>
#include <sys/types.h>
#include <sys/sysctl.h>
#include <sys/user.h>
#include <fcntl.h>
#include <iostream>
#ifdef __FreeBSD__
#include <libutil.h>
#elif defined(__APPLE__)
#include <libproc.h>
#endif

namespace Animia::Unix {

/* this is a cleaned up version of a function from... Apple?
   ...anyway, what it essentially does is gets the size and stuff from
   sysctl() and reserves the space in a vector to store the PIDs */
std::vector<int> get_all_pids() {
    std::vector<int> ret;
    struct kinfo_proc* result = NULL;
    size_t             length = 0;
    static const int   name[] = { CTL_KERN, KERN_PROC, KERN_PROC_ALL, 0 };

    /* get appropriate length from sysctl() */
    sysctl((int*)name, (sizeof(name) / sizeof(*name)) - 1, NULL, &length, NULL, 0);

    result = (struct kinfo_proc*)malloc(length);
    if (result == NULL)
        return std::vector<int>();

    /* actually get our results */
    if (sysctl((int*)name, (sizeof(name) / sizeof(*name)) - 1, result, &length, NULL, 0) == ENOMEM) {
        assert(result != NULL);
        free(result);
        throw std::bad_alloc();
    }

    /* add pids to our vector */
    ret.reserve(length/sizeof(*result));
    for (int i = 0; i < length/sizeof(*result); i++)
        ret.push_back(result[i].kp_proc.p_pid);

    return ret;
}

std::string get_process_name(int pid) {
    std::string ret;
#ifdef __FreeBSD__
    struct kinfo_proc* proc = kinfo_getproc(pid);
    if (!proc) {
        return "";
    ret = proc->ki_comm;
    free(proc);
#elif defined(__APPLE__)
    struct proc_bsdinfo proc;

    int st = proc_pidinfo(pid, PROC_PIDTBSDINFO, 0, &proc, PROC_PIDTBSDINFO_SIZE);
    if (st != PROC_PIDTBSDINFO_SIZE)
        return "";
    return proc.pbi_comm;
#endif
    return ret;
}

std::vector<std::string> get_open_files(int pid) {
    // initialize buffer
    std::vector<std::string> ret;
    int bufsz = proc_pidinfo(pid, PROC_PIDLISTFDS, 0, NULL, 0);
    struct proc_fdinfo *info = (struct proc_fdinfo *)malloc(bufsz);
    proc_pidinfo(pid, PROC_PIDLISTFDS, 0, info, bufsz);

    // iterate over stuff
    for (int i = 0; i < bufsz/sizeof(info[0]); i++) {
        if (info[i].proc_fdtype == PROX_FDTYPE_VNODE) {
            struct vnode_fdinfowithpath vnodeInfo;
            proc_pidfdinfo(pid, info[i].proc_fd, PROC_PIDFDVNODEPATHINFO, &vnodeInfo, PROC_PIDFDVNODEPATHINFO_SIZE);
	    ret.push_back(vnodeInfo.pvip.vip_path);
        }
    }
    return ret;
}

std::unordered_map<int, std::vector<std::string>> get_all_open_files() {
    std::unordered_map<int, std::vector<std::string>> map;
    std::vector<int> pids = get_all_pids();
    for (int i: pids) {
        map[i] = get_open_files(i);
    }
    return map;
}

}
