#include "gui/dialog/information.h"
#include "core/anime.h"
#include "core/anime_db.h"
#include "core/array.h"
#include "core/strings.h"
#include "gui/pages/anime_list.h"
#include "gui/translate/anime.h"
#include "gui/widgets/optional_date.h"
#include "gui/widgets/text.h"
#include "gui/window.h"
#include <QCheckBox>
#include <QComboBox>
#include <QDateEdit>
#include <QDebug>
#include <QDialogButtonBox>
#include <QLineEdit>
#include <QPlainTextEdit>
#include <QSpinBox>
#include <QStringList>
#include <QTextStream>
#include <QVBoxLayout>
#include <functional>

/* TODO: Taiga disables rendering of the tab widget entirely when the anime is not part of a list,
   which sucks. Think of a better way to implement this later. */
void InformationDialog::SaveData() {
	Anime::Anime& anime = Anime::db.items[id];
	anime.SetUserProgress(progress);
	anime.SetUserScore(score);
	anime.SetUserIsRewatching(rewatching);
	anime.SetUserStatus(status);
	anime.SetUserNotes(notes);
	anime.SetUserDateStarted(started);
	anime.SetUserDateCompleted(completed);
}

InformationDialog::InformationDialog(const Anime::Anime& anime, std::function<void()> accept, QWidget* parent)
    : QDialog(parent) {
	setFixedSize(842, 613);
	setWindowTitle(tr("Anime Information"));
	setWindowFlags(Qt::Dialog | Qt::CustomizeWindowHint | Qt::WindowTitleHint | Qt::WindowCloseButtonHint);

	{
		QPalette pal;
		pal.setColor(QPalette::Window, Qt::white);
		setPalette(pal);
	}

	QWidget* widget = new QWidget(this);

	/* "sidebar", includes... just the anime image :) */
	QWidget* sidebar = new QWidget(widget);
	sidebar->setFixedWidth(175);

	/* main widget */
	QWidget* main_widget = new QWidget(widget);

	{
		QPalette pal;
		pal.setColor(QPalette::Window, Qt::white);
		main_widget->setPalette(pal);
	}

	main_widget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

	id = anime.GetId();
	/* anime title header text */
	TextWidgets::Paragraph* anime_title =
	    new TextWidgets::Paragraph(QString::fromUtf8(anime.GetUserPreferredTitle().c_str()), main_widget);
	anime_title->setReadOnly(true);
	anime_title->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	anime_title->setWordWrapMode(QTextOption::NoWrap);
	anime_title->setFrameShape(QFrame::NoFrame);
	anime_title->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Maximum);
	anime_title->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	anime_title->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

	{
		QFont font(anime_title->font());
		font.setPointSize(12);
		anime_title->setFont(font);
	}

	{
		QPalette pal;
		pal.setColor(QPalette::Window, Qt::transparent);
		pal.setColor(QPalette::WindowText, Qt::blue);
	}

	/* tabbed widget */
	QTabWidget* tabbed_widget = new QTabWidget(main_widget);
	tabbed_widget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);

	/* main info tab */
	QWidget* main_information_widget = new QWidget(tabbed_widget);
	main_information_widget->setLayout(new QVBoxLayout);

	/* alt titles */
	main_information_widget->layout()->addWidget(new TextWidgets::SelectableTextParagraph(
	    tr("Alternative titles"), QString::fromUtf8(Strings::Implode(anime.GetTitleSynonyms(), ", ").c_str()),
	    main_information_widget));

	/* details */
	QString details_data;
	QTextStream details_data_s(&details_data);
	details_data_s << Translate::ToString(anime.GetFormat()).c_str() << "\n"
	               << anime.GetEpisodes() << "\n"
	               << Translate::ToString(anime.GetUserStatus()).c_str() << "\n"
	               << Translate::ToString(anime.GetSeason()).c_str() << " " << anime.GetAirDate().GetYear() << "\n"
	               << Strings::Implode(anime.GetGenres(), ", ").c_str() << "\n"
	               << anime.GetAudienceScore() << "%";
	main_information_widget->layout()->addWidget(
	    new TextWidgets::LabelledTextParagraph(tr("Details"), tr("Type:\nEpisodes:\nStatus:\nSeason:\nGenres:\nScore:"),
	                                           details_data, main_information_widget));

	/* synopsis */
	TextWidgets::SelectableTextParagraph* synopsis = new TextWidgets::SelectableTextParagraph(
	    tr("Synopsis"), QString::fromUtf8(anime.GetSynopsis().c_str()), main_information_widget);

	synopsis->GetParagraph()->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding);
	main_information_widget->layout()->addWidget(synopsis);

	QWidget* settings_widget = new QWidget(tabbed_widget);
	settings_widget->setLayout(new QVBoxLayout);
	settings_widget->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum);

	settings_widget->layout()->addWidget(new TextWidgets::Header(tr("Anime list"), settings_widget));

	QWidget* sg_anime_list_content = new QWidget(settings_widget);
	settings_widget->layout()->addWidget(sg_anime_list_content);
	sg_anime_list_content->setLayout(new QVBoxLayout);
	sg_anime_list_content->layout()->setSpacing(5);
	sg_anime_list_content->layout()->setContentsMargins(12, 0, 0, 0);

/* these macros make this a lot easier to edit */
#define LAYOUT_HORIZ_SPACING 25
#define LAYOUT_VERT_SPACING  5
#define LAYOUT_ITEM_WIDTH    175
/* Creates a subsection with a width of 175 */
#define CREATE_SUBSECTION(x) \
	{ \
		QWidget* subsection = new QWidget(section); \
		subsection->setLayout(new QVBoxLayout); \
		subsection->setFixedWidth(LAYOUT_ITEM_WIDTH); \
		subsection->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred); \
		subsection->layout()->setSpacing(LAYOUT_VERT_SPACING); \
		subsection->layout()->setContentsMargins(0, 0, 0, 0); \
		x; \
		layout->addWidget(subsection); \
	}
/* Creates a section in the parent `a` */
#define CREATE_SECTION(a, x) \
	{ \
		QWidget* section = new QWidget(a); \
		QHBoxLayout* layout = new QHBoxLayout(section); \
		layout->setSpacing(LAYOUT_HORIZ_SPACING); \
		layout->setContentsMargins(0, 0, 0, 0); \
		x; \
		layout->addStretch(); \
		a->layout()->addWidget(section); \
	}
/* Creates a subsection that takes up whatever space is necessary */
#define CREATE_FULL_WIDTH_SUBSECTION(x) \
	{ \
		QWidget* subsection = new QWidget(section); \
		subsection->setLayout(new QVBoxLayout); \
		subsection->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred); \
		subsection->layout()->setSpacing(LAYOUT_VERT_SPACING); \
		subsection->layout()->setContentsMargins(0, 0, 0, 0); \
		x; \
		layout->addWidget(subsection); \
	}
/* Creates a section in the parent `a` */
#define CREATE_FULL_WIDTH_SECTION(a, x) \
	{ \
		QWidget* section = new QWidget(a); \
		QHBoxLayout* layout = new QHBoxLayout(section); \
		layout->setSpacing(LAYOUT_HORIZ_SPACING); \
		layout->setContentsMargins(0, 0, 0, 0); \
		x; \
		a->layout()->addWidget(section); \
	}

	CREATE_SECTION(sg_anime_list_content, {
		/* Episodes watched section */
		CREATE_SUBSECTION({
			subsection->layout()->addWidget(new QLabel(tr("Episodes watched:"), subsection));

			QSpinBox* spin_box = new QSpinBox(subsection);
			connect(spin_box, QOverload<int>::of(&QSpinBox::valueChanged), this, [this](int i) { progress = i; });
			spin_box->setRange(0, anime.GetEpisodes());
			spin_box->setSingleStep(1);
			spin_box->setValue(progress = anime.GetUserProgress());
			subsection->layout()->addWidget(spin_box);
		});
		CREATE_SUBSECTION({
			subsection->layout()->addWidget(new QLabel(tr(" "), subsection));

			QCheckBox* checkbox = new QCheckBox(tr("Rewatching"));
			connect(checkbox, QOverload<int>::of(&QCheckBox::stateChanged), this,
			        [this](int state) { rewatching = (state == Qt::Checked); });
			checkbox->setCheckState(anime.GetUserIsRewatching() ? Qt::Checked : Qt::Unchecked);
			subsection->layout()->addWidget(checkbox);
		});
	});
	CREATE_SECTION(sg_anime_list_content, {
		/* Status & score section */
		CREATE_SUBSECTION({
			subsection->layout()->addWidget(new QLabel(tr("Status:"), subsection));

			QStringList string_list;
			for (unsigned int i = 0; i < ARRAYSIZE(Anime::ListStatuses); i++)
				string_list.append(QString::fromStdString(Translate::ToString(Anime::ListStatuses[i])));

			QComboBox* combo_box = new QComboBox(subsection);
			combo_box->addItems(string_list);
			connect(combo_box, QOverload<int>::of(&QComboBox::currentIndexChanged), this,
			        [this](int i) { status = Anime::ListStatuses[i]; });
			combo_box->setCurrentIndex(static_cast<int>(status = anime.GetUserStatus()) - 1);
			subsection->layout()->addWidget(combo_box);
		});
		CREATE_SUBSECTION({
			subsection->layout()->addWidget(new QLabel(tr("Score:"), subsection));

			QSpinBox* spin_box = new QSpinBox(subsection);
			connect(spin_box, QOverload<int>::of(&QSpinBox::valueChanged), this, [this](int i) { score = i; });
			spin_box->setRange(0, 100);
			spin_box->setSingleStep(5);
			spin_box->setValue(score = anime.GetUserScore());
			subsection->layout()->addWidget(spin_box);
		});
	});
	CREATE_FULL_WIDTH_SECTION(sg_anime_list_content, {
		/* Notes section */
		CREATE_FULL_WIDTH_SUBSECTION({
			subsection->layout()->addWidget(new QLabel(tr("Notes:"), subsection));

			QLineEdit* line_edit = new QLineEdit(subsection);
			connect(line_edit, &QLineEdit::textChanged, this, [this](const QString& text) {
				/* this sucks but I don't really want to implement anything smarter :) */
				notes = text.toStdString();
			});
			line_edit->setText(QString::fromStdString(notes = anime.GetUserNotes()));
			line_edit->setPlaceholderText(tr("Enter your notes about this anime"));
			subsection->layout()->addWidget(line_edit);
		});
	});
	CREATE_SECTION(sg_anime_list_content, {
		/* Dates section */
		CREATE_SUBSECTION({
			subsection->layout()->addWidget(new QLabel(tr("Date started:"), subsection));

			OptionalDate* date = new OptionalDate(true, subsection);
			connect(date, &OptionalDate::DataChanged, this,
			        [this](bool enabled, Date date) { started = (enabled) ? date : Date(); });
			started = anime.GetUserDateStarted();
			if (!started.IsValid()) {
				date->SetEnabled(false);
				started = anime.GetAirDate();
			}
			date->SetDate(started);
			subsection->layout()->addWidget(date);
		});
		CREATE_SUBSECTION({
			subsection->layout()->addWidget(new QLabel(tr("Date completed:"), subsection));

			OptionalDate* date = new OptionalDate(true, subsection);
			connect(date, &OptionalDate::DataChanged, this,
			        [this](bool enabled, Date date) { completed = (enabled) ? date : Date(); });
			completed = anime.GetUserDateCompleted();
			if (!completed.IsValid()) {
				date->SetEnabled(false);
				completed = anime.GetAirDate();
			}
			date->SetDate(completed);
			subsection->layout()->addWidget(date);
		});
	});

	settings_widget->layout()->addWidget(new TextWidgets::Header(tr("Local settings"), settings_widget));

	QWidget* sg_local_content = new QWidget(settings_widget);
	settings_widget->layout()->addWidget(sg_local_content);
	sg_local_content->setLayout(new QVBoxLayout);
	sg_local_content->layout()->setSpacing(5);
	sg_local_content->layout()->setContentsMargins(12, 0, 0, 0);

	CREATE_FULL_WIDTH_SECTION(sg_local_content, {
		/* Alternative titles */
		CREATE_FULL_WIDTH_SUBSECTION({
			subsection->layout()->addWidget(new QLabel(tr("Alternative titles:"), subsection));

			QLineEdit* line_edit = new QLineEdit(QString::fromStdString(anime.GetUserNotes()), subsection);
			line_edit->setPlaceholderText(
			    tr("Enter alternative titles here, separated by a semicolon (i.e. Title 1; Title 2)"));
			subsection->layout()->addWidget(line_edit);

			QCheckBox* checkbox = new QCheckBox(tr("Use the first alternative title to search for torrents"));
			subsection->layout()->addWidget(checkbox);
		});
	});
#undef CREATE_SECTION
#undef CREATE_SUBSECTION
#undef CREATE_FULL_WIDTH_SECTION
#undef CREATE_FULL_WIDTH_SUBSECTION

	static_cast<QBoxLayout*>(settings_widget->layout())->addStretch();

	tabbed_widget->addTab(main_information_widget, tr("Main information"));
	tabbed_widget->addTab(settings_widget, tr("My list and settings"));

	QVBoxLayout* main_layout = new QVBoxLayout;
	main_layout->addWidget(anime_title);
	main_layout->addWidget(tabbed_widget);
	main_layout->setContentsMargins(0, 0, 0, 0);
	main_widget->setLayout(main_layout);

	QHBoxLayout* layout = new QHBoxLayout;
	layout->addWidget(sidebar);
	layout->addWidget(main_widget);
	widget->setLayout(layout);

	QDialogButtonBox* button_box = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel, this);
	connect(button_box, &QDialogButtonBox::accepted, this, [this, accept] {
		SaveData();
		accept();
		QDialog::accept();
	});
	connect(button_box, &QDialogButtonBox::rejected, this, &QDialog::reject);

	QVBoxLayout* buttons_layout = new QVBoxLayout;
	buttons_layout->addWidget(widget);
	buttons_layout->addWidget(button_box, 0, Qt::AlignBottom);
	setLayout(buttons_layout);
}

#include "gui/dialog/moc_information.cpp"
