#include "gui/pages/statistics.h"
#include "core/anime_db.h"
#include "core/session.h"
#include "core/strings.h"
#include "gui/pages/anime_list.h"
#include "gui/widgets/graph.h"
#include "gui/widgets/text.h"

#include <QString>
#include <QTextDocument>
#include <QTextStream>
#include <QTimer>
#include <QVBoxLayout>
#include <QWidget>

#include <cmath>
#include <sstream>

enum class TimeUnits {
	SECONDS,
	MINUTES
};

StatisticsPage::StatisticsPage(QWidget* parent) : QFrame(parent) {
	QVBoxLayout* layout = new QVBoxLayout(this);

	setFrameShape(QFrame::Box);
	setFrameShadow(QFrame::Sunken);

	setAutoFillBackground(true);

	_anime_list.reset(new TextWidgets::LabelledSection(
	    tr("Anime list"),
	    tr("Anime count:\nEpisode count:\nTime spent watching:\nTime to complete:\nAverage score:\nScore deviation:"),
	    "", this));
	layout->addWidget(_anime_list.get());

	{
		QWidget* score_dist_widget = new QWidget(this);
		QVBoxLayout* score_dist_layout = new QVBoxLayout(score_dist_widget);

		score_dist_layout->addWidget(new TextWidgets::Header(tr("Score distribution"), score_dist_widget));

		/* Ew */
		{
			QWidget* score_graph_parent = new QWidget(score_dist_widget);
			QVBoxLayout* score_parent_layout = new QVBoxLayout(score_graph_parent);

			_score_distribution_graph.reset(new Graph<int>(score_graph_parent));
			score_parent_layout->addWidget(_score_distribution_graph.get());

			score_parent_layout->setSpacing(0);
			score_parent_layout->setContentsMargins(12, 0, 0, 0);

			score_dist_layout->addWidget(score_graph_parent);
		}

		score_dist_layout->setContentsMargins(0, 0, 0, 0);

		layout->addWidget(score_dist_widget);
	}

	_application.reset(new TextWidgets::LabelledSection(tr("Minori"), tr("Uptime:\nRequests made:"), "\n\n", this));
	layout->addWidget(_application.get());

	layout->addStretch();

	QTimer* timer = new QTimer(this);
	connect(timer, &QTimer::timeout, this, [this] {
		if (isVisible())
			UpdateStatistics();
	});
	timer->start(1000); // update statistics every second
}

void StatisticsPage::showEvent(QShowEvent*) {
	UpdateStatistics();
}

/*  [in] enum TimeUnits unit:
 *       which unit to stop on
 *  [in] int amount:
 *       amount of units to parse
 *  [in, defaults to 1.0] double unit_in_seconds:
 *       equivalent of one of 'amount' in seconds, e.g. minutes would be 60.0
 */
static std::string TimeToDateString(TimeUnits unit, int amount, double unit_in_seconds = 1.0) {
	/* avoid calculating this twice */
	const double years_conv = (31556952.0 / unit_in_seconds);
	const double months_conv = (2629746.0 / unit_in_seconds);
	const double days_conv = (86400.0 / unit_in_seconds);
	const double hours_conv = (3600.0 / unit_in_seconds);
	const double minutes_conv = (60.0 / unit_in_seconds);
	const double seconds_conv = (1.0 / unit_in_seconds);

	const int years = amount / years_conv;
	const int months = std::fmod(amount, years_conv) / months_conv;
	const int days = std::fmod(amount, months_conv) / days_conv;
	const int hours = std::fmod(amount, days_conv) / hours_conv;
	const int minutes = std::fmod(amount, hours_conv) / minutes_conv;
	const int seconds = std::fmod(amount, minutes_conv) / seconds_conv;

	const auto add_time_segment = [](std::ostringstream& str, int amount, const std::string_view& singular,
	                                 const std::string_view& plural, bool always = false) {
		if (amount > 0 || always)
			str << amount << ((amount == 1) ? singular : plural);
	};

	std::ostringstream string;
	add_time_segment(string, years, " year ", " years ");
	add_time_segment(string, months, " month ", " months ");
	add_time_segment(string, days, " day ", " days ");
	add_time_segment(string, hours, " hour ", " hours ");

	if (unit == TimeUnits::MINUTES) {
		add_time_segment(string, minutes, " minute", " minutes", true);
		return string.str();
	} else {
		add_time_segment(string, minutes, " minute ", " minutes ");
	}

	add_time_segment(string, seconds, " second", " seconds", true);
	return string.str();
}

inline int GetTotalWithScore(const int score) {
	int count = 0;
	for (const auto& item : Anime::db.items)
		if (item.second.IsInUserList() && item.second.GetUserScore() == score)
			count++;
	return count;
}

void StatisticsPage::UpdateStatistics() {
	/* Anime list */
	QString string = "";
	QTextStream ts(&string);
	ts << Anime::db.GetTotalAnimeAmount() << '\n';
	ts << Anime::db.GetTotalEpisodeAmount() << '\n';
	ts << Strings::ToQString(TimeToDateString(TimeUnits::MINUTES, Anime::db.GetTotalWatchedAmount(), 60.0)) << '\n';
	ts << Strings::ToQString(TimeToDateString(TimeUnits::MINUTES, Anime::db.GetTotalPlannedAmount(), 60.0)) << '\n';
	ts << Anime::db.GetAverageScore() << '\n';
	ts << Anime::db.GetScoreDeviation();
	_anime_list->GetParagraph()->SetText(string);

	_score_distribution_graph->Clear();
	for (int i = 10; i <= 100; i += 10)
		_score_distribution_graph->AddItem(i, GetTotalWithScore(i));

	string = "";
	ts << Strings::ToQString(TimeToDateString(TimeUnits::SECONDS, session.uptime() / 1000)) << '\n';
	ts << session.GetRequests();
	/* Application */
	// UiUtils::SetPlainTextEditData(application_data, QString::number(session.uptime() / 1000));
	_application->GetParagraph()->SetText(string);
}
