#ifdef WIN32
#	include <shlobj.h>
#elif defined(MACOSX)
#	include "sys/osx/filesystem.h"
#elif defined(__linux__)
#	include <pwd.h>
#	include <sys/types.h>
#endif

#ifdef WIN32
#	define DELIM "\\"
#else
#	define DELIM "/"
#	include <errno.h>
#	include <unistd.h>
#	include <sys/stat.h>
#endif

#include "core/config.h"
#include "core/filesystem.h"
#include <limits.h>

namespace Filesystem {

bool CreateDirectories(std::string path) {
	std::string temp = "";
	size_t start;
	size_t end = 0;

	while ((start = path.find_first_not_of(DELIM, end)) != std::string::npos) {
		end = path.find(DELIM, start);
		temp.append(path.substr(start, end - start));
#ifdef WIN32
		if (!CreateDirectoryA(temp.c_str(), NULL) && GetLastError() == ERROR_PATH_NOT_FOUND)
			/* ERROR_PATH_NOT_FOUND should NOT happen here */
			return false;
#else
		if (mkdir(temp.c_str(), 0755))
			return false;
#endif
		temp.append(DELIM);
	}
	return true;
}

bool Exists(std::string path) {
#if WIN32
	return GetFileAttributes(path.c_str()) != INVALID_FILE_ATTRIBUTES;
#else
	struct stat st;
	return stat(path.c_str(), &st) == 0;
#endif
}

std::string GetParentDirectory(std::string path) {
	size_t pos = 0;
	pos = path.find_last_of(DELIM);
	return path.substr(0, pos);
}

std::string GetConfigPath(void) {
	std::string ret = "";
#ifdef WIN32
	char buf[PATH_MAX + 1];
	if (SHGetFolderPathAndSubDir(NULL, CSIDL_APPDATA | CSIDL_FLAG_CREATE, NULL, 0, CONFIG_DIR, buf) == S_OK) {
		ret += buf;
		ret += DELIM CONFIG_NAME;
	}
#elif defined(MACOSX)
	/* pass all of our problems to objc++ code */
	ret += osx::GetApplicationSupportDirectory();
	ret += DELIM CONFIG_DIR DELIM CONFIG_NAME;
#else // just assume POSIX
	if (getenv("HOME") != NULL)
		ret += getenv("HOME");
#	ifdef __linux__
	else
		ret += getpwuid(getuid())->pw_dir;
#	endif // __linux__
	if (!ret.empty())
		ret += DELIM ".config" DELIM CONFIG_DIR DELIM CONFIG_NAME;
#endif     // !WIN32 && !MACOSX
	return ret;
}

} // namespace Filesystem
