/*
 * util/win32.cc: utility functions only useful on windows
 */
#include "animone/util/win32.h"

#include <shlobj.h>  /* SHGetKnownFolderPath */
#include <subauth.h> /* UNICODE_STRING */
#include <windows.h>

namespace animone::internal::win32 {

std::string ToUtf8String(const std::wstring& string) {
	if (string.empty())
		return std::string();

	long size = ::WideCharToMultiByte(CP_UTF8, 0, string.c_str(), string.length(), nullptr, 0, nullptr, nullptr);
	std::string ret(size, '\0');
	::WideCharToMultiByte(CP_UTF8, 0, string.c_str(), string.length(), &ret.front(), ret.length(), nullptr, nullptr);
	return ret;
}

std::string ToUtf8String(const UNICODE_STRING& string) {
	const auto wctomb = [&string](LPSTR out, int size) -> int {
		return ::WideCharToMultiByte(CP_UTF8, 0, string.Buffer, string.Length, out, size, nullptr, nullptr);
	};

	if (string.Length <= 0)
		return std::string();

	long size = wctomb(nullptr, 0);
	std::string ret(size, '\0');
	wctomb(&ret.front(), ret.length());
	return ret;
}

std::wstring ToWstring(const std::string& string) {
	const auto mbtowc = [&string](LPWSTR out, int size) -> int {
		return ::MultiByteToWideChar(CP_UTF8, 0, string.c_str(), string.length(), out, size);
	};

	if (string.empty())
		return std::wstring();

	long size = mbtowc(nullptr, 0);
	std::wstring ret(size, L'\0');
	mbtowc(&ret.front(), ret.length());
	return ret;
}

std::wstring GetProcessPath(DWORD process_id) {
	// If we try to open a SYSTEM process, this function fails and the last error
	// code is ERROR_ACCESS_DENIED.
	//
	// Note that if we requested PROCESS_QUERY_INFORMATION access right instead
	// of PROCESS_QUERY_LIMITED_INFORMATION, this function would fail when used
	// to open an elevated process.
	Handle process_handle(::OpenProcess(PROCESS_QUERY_LIMITED_INFORMATION, FALSE, process_id));

	if (!process_handle)
		return std::wstring();

	std::wstring buffer(MAX_PATH, L'\0');
	DWORD buf_size = buffer.length();

	// Note that this function requires Windows Vista or above. You may use
	// GetProcessImageFileName or GetModuleFileNameEx on earlier versions.
	if (!::QueryFullProcessImageNameW(process_handle.get(), 0, &buffer.front(), &buf_size))
		return std::wstring();

	buffer.resize(buf_size);
	return buffer;
}

std::wstring GetFileNameFromPath(const std::wstring& path) {
	const auto pos = path.find_last_of(L"/\\");
	return pos != std::wstring::npos ? path.substr(pos + 1) : path;
}

std::wstring GetFileNameWithoutExtension(const std::wstring& filename) {
	const auto pos = filename.find_last_of(L".");
	return pos != std::wstring::npos ? filename.substr(0, pos) : filename;
}

static std::wstring GetSystemDirectory() {
	PWSTR path_wch;
	SHGetKnownFolderPath(FOLDERID_Windows, 0, NULL, &path_wch);
	std::wstring path_wstr(path_wch);
	CoTaskMemFree(path_wch);
	return path_wstr;
}

bool IsSystemDirectory(const std::string& path) {
	return IsSystemDirectory(ToWstring(path));
}

bool IsSystemDirectory(std::wstring path) {
	::CharUpperBuffW(&path.front(), path.length());

	std::wstring windir = GetSystemDirectory();
	::CharUpperBuffW(&windir.front(), windir.length());

	return path.find(windir) == 4;
}

} // namespace animone::internal::win32
