#include "core/date.h"
#include "core/json.h"
#include <QDate>
#include <cstdint>
#include <tuple>

/* An implementation of AniList's "fuzzy date" */

#define MIN(A, B)                                                                                                      \
	({                                                                                                                 \
		__typeof__(A) __a = (A);                                                                                       \
		__typeof__(B) __b = (B);                                                                                       \
		__a < __b ? __a : __b;                                                                                         \
	})
#define MAX(A, B)                                                                                                      \
	({                                                                                                                 \
		__typeof__(A) __a = (A);                                                                                       \
		__typeof__(B) __b = (B);                                                                                       \
		__a < __b ? __b : __a;                                                                                         \
	})

#define CLAMP(x, low, high)                                                                                            \
	({                                                                                                                 \
		__typeof__(x) __x = (x);                                                                                       \
		__typeof__(low) __low = (low);                                                                                 \
		__typeof__(high) __high = (high);                                                                              \
		__x > __high ? __high : (__x < __low ? __low : __x);                                                           \
	})

Date::Date() {
}

Date::Date(int32_t y) {
	year = std::make_unique<int32_t>(MAX(0, y));
}

Date::Date(int32_t y, int8_t m, int8_t d) {
	year = std::make_unique<int32_t>(MAX(0, y));
	month = std::make_unique<int8_t>(CLAMP(m, 1, 12));
	day = std::make_unique<int8_t>(CLAMP(d, 1, 31));
}

void Date::VoidYear() {
	year.reset();
}

void Date::VoidMonth() {
	month.reset();
}

void Date::VoidDay() {
	day.reset();
}

void Date::SetYear(int32_t y) {
	year = std::make_unique<int32_t>(MAX(0, y));
}

void Date::SetMonth(int8_t m) {
	month = std::make_unique<int8_t>(CLAMP(m, 1, 12));
}

void Date::SetDay(int8_t d) {
	day = std::make_unique<int8_t>(CLAMP(d, 1, 31));
}

int32_t Date::GetYear() const {
	int32_t* ptr = year.get();
	if (ptr != nullptr)
		return *year;
	return -1;
}

int8_t Date::GetMonth() const {
	int8_t* ptr = month.get();
	if (ptr != nullptr)
		return *month;
	return -1;
}

int8_t Date::GetDay() const {
	int8_t* ptr = day.get();
	if (ptr != nullptr)
		return *day;
	return -1;
}

bool Date::operator<(const Date& other) const {
	int o_y = other.GetYear(), o_m = other.GetMonth(), o_d = other.GetDay();
	return std::tie(*year, *month, *day) < std::tie(o_y, o_m, o_d);
}

bool Date::operator>(const Date& other) const {
	return other < (*this);
}

bool Date::operator<=(const Date& other) const {
	return !((*this) > other);
}

bool Date::operator>=(const Date& other) const {
	return !((*this) < other);
}

QDate Date::GetAsQDate() const {
	/* QDates don't (yet) support "missing" values */
	if (year.get() && month.get() && day.get())
		return QDate(*year, *month, *day);
	else return QDate();
}

nlohmann::json Date::GetAsAniListJson() const {
	nlohmann::json result;
	if (year.get())
		result.insert(result.end(), {"year", *year});
	else
		result.insert(result.end(), {"year", nullptr});
	if (month.get())
		result.insert(result.end(), {"month", *month});
	else
		result.insert(result.end(), {"month", nullptr});
	if (day.get())
		result.insert(result.end(), {"day", *day});
	else
		result.insert(result.end(), {"day", nullptr});
	return result;
}
