#include <QApplication>
#include <QFile>
#include <QTextStream>
#include "core/config.h"
#include "core/session.h"
#ifdef MACOSX
#	include "sys/osx/dark_theme.h"
#else
#	include "sys/win32/dark_theme.h"
#endif

namespace DarkTheme {

bool IsInDarkMode() {
	if (session.config.theme != Themes::OS)
		return (session.config.theme == Themes::DARK);
#ifdef MACOSX
	if (osx::DarkThemeAvailable()) {
		if (osx::IsInDarkTheme()) {
			return true;
		} else {
			return false;
		}
	}
#elif defined(WIN32)
	if (win32::DarkThemeAvailable()) {
		if (win32::IsInDarkTheme()) {
			return true;
		} else {
			return false;
		}
	}
#endif
	return (session.config.theme == Themes::DARK);
}

/* this function is private, and should stay that way */
void SetStyleSheet(enum Themes theme) {
	switch (theme) {
		case Themes::DARK: {
			QFile f(":qdarkstyle/dark/darkstyle.qss");
			if (!f.exists())
				return; // fail
			f.open(QFile::ReadOnly | QFile::Text);
			QTextStream ts(&f);
			qApp->setStyleSheet(ts.readAll());
			break;
		}
		default: qApp->setStyleSheet(""); break;
	}
}

void SetToDarkTheme() {
	/* macOS >= 10.14 has its own global dark theme,
	   use it :) */
#if MACOSX
	if (osx::DarkThemeAvailable())
		osx::SetToDarkTheme();
	else
#endif
	SetStyleSheet(Themes::DARK);
}

void SetToLightTheme() {
#if MACOSX
	if (osx::DarkThemeAvailable())
		osx::SetToLightTheme();
	else
#endif
		SetStyleSheet(Themes::LIGHT);
}

enum Themes GetCurrentOSTheme() {
#if MACOSX
	if (osx::DarkThemeAvailable())
		return osx::IsInDarkTheme() ? Themes::DARK : Themes::LIGHT;
#elif defined(WIN32)
	if (win32::DarkThemeAvailable())
		return win32::IsInDarkTheme() ? Themes::DARK : Themes::LIGHT;
#endif
	/* Currently OS detection only supports Windows and macOS.
	   Please don't be shy if you're willing to port it to other OSes
	   (or desktop environments, or window managers) */
	return Themes::LIGHT;
}

void SetTheme(enum Themes theme) {
	switch (theme) {
		case Themes::LIGHT: SetToLightTheme(); break;
		case Themes::DARK: SetToDarkTheme(); break;
		case Themes::OS:
			if (GetCurrentOSTheme() == Themes::LIGHT)
				SetToLightTheme();
			else
				SetToDarkTheme();
			break;
	}
}

} // namespace DarkTheme
