#include "gui/dialog/settings.h"
#include "core/session.h"
#include "gui/widgets/sidebar.h"
#include "gui/widgets/text.h"
#include <QDialogButtonBox>
#include <QHBoxLayout>
#include <QLabel>
#include <QStackedWidget>
#include <QVBoxLayout>
#include <QWidget>
#ifdef WIN32
#	include "sys/win32/dark_theme.h"
#endif

SettingsPage::SettingsPage(QWidget* parent, QString title)
	: QWidget(parent)
	, page_title_(title)
	, layout_(this) {
	setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding);

	page_title_.setWordWrap(false);
	page_title_.setFrameShape(QFrame::Panel);
	page_title_.setFrameShadow(QFrame::Sunken);

	{
		QFont fnt(page_title_.font());
		fnt.setPixelSize(12);
		fnt.setWeight(QFont::Bold);
		page_title_.setFont(fnt);
	}

	{
		QPalette pal(page_title_.palette());
		pal.setColor(QPalette::Window, QColor(0xAB, 0xAB, 0xAB));
		pal.setColor(QPalette::WindowText, Qt::white);
		page_title_.setPalette(pal);
	}

	page_title_.setAutoFillBackground(true);

	page_title_.setFixedHeight(23);
	page_title_.setAlignment(Qt::AlignVCenter | Qt::AlignLeft);
	page_title_.setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Fixed);

	tab_widget_.setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding);

	layout_.setContentsMargins(0, 0, 0, 0);
	layout_.addWidget(&page_title_);
	layout_.addWidget(&tab_widget_);
}

void SettingsPage::SetTitle(QString title) {
	page_title_.setText(title);
}

void SettingsPage::AddTab(QWidget* tab, QString title) {
	tab_widget_.addTab(tab, title);
}

void SettingsPage::SaveInfo() {
	// no-op... child classes will implement this
}

void SettingsDialog::OnOK() {
	services_page_.SaveInfo();
	library_page_.SaveInfo();
	application_page_.SaveInfo();
	recognition_page_.SaveInfo();
	torrents_page_.SaveInfo();

	QDialog::accept();
}

void SettingsDialog::showEvent(QShowEvent* event) {
	QDialog::showEvent(event);
#ifdef WIN32
	win32::SetTitleBarsToBlack(session.config.theme.IsInDarkTheme());
#endif
}

SettingsDialog::SettingsDialog(QWidget* parent)
	: QDialog(parent)
	, layout_(this)
	, widget_layout_(&widget_)
	, button_box_(QDialogButtonBox::Ok | QDialogButtonBox::Cancel) {
	setFixedSize(755, 566);
	setWindowTitle(tr("Settings"));
	setWindowFlags(Qt::Dialog | Qt::CustomizeWindowHint | Qt::WindowTitleHint | Qt::WindowCloseButtonHint);

	{
		widget_.setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding);

		{
			sidebar_.setCurrentItem(sidebar_.AddItem(tr("Services"), SideBar::CreateIcon(":/icons/24x24/globe.png")));
			sidebar_.AddItem(tr("Library"), SideBar::CreateIcon(":/icons/24x24/inbox-film.png"));
			sidebar_.AddItem(tr("Application"), SideBar::CreateIcon(":/icons/24x24/application-sidebar-list.png"));
			sidebar_.AddItem(tr("Recognition"), SideBar::CreateIcon(":/icons/24x24/question.png"));
			// sidebar->AddItem(tr("Sharing"), SideBar::CreateIcon(":/icons/24x24/megaphone.png"));
			sidebar_.AddItem(tr("Torrents"), SideBar::CreateIcon(":/icons/24x24/feed.png"));
			// sidebar->AddItem(tr("Advanced"), SideBar::CreateIcon(":/icons/24x24/gear.png"));

			sidebar_.setIconSize(QSize(24, 24));
			sidebar_.setFrameShape(QFrame::Box);

			sidebar_.SetBackgroundTransparent(false);
			sidebar_.setBackgroundRole(QPalette::Base);

			sidebar_.setFixedWidth(158);
			sidebar_.setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding);
			widget_layout_.addWidget(&sidebar_);
		}

		{
			stacked_.addWidget(&services_page_);
			stacked_.addWidget(&library_page_);
			stacked_.addWidget(&application_page_);
			stacked_.addWidget(&recognition_page_);
			stacked_.addWidget(&torrents_page_);
			stacked_.setCurrentIndex(0);

			connect(&sidebar_, &QListWidget::currentRowChanged, &stacked_, &QStackedWidget::setCurrentIndex);

			widget_layout_.addWidget(&stacked_);
		}

		widget_layout_.setContentsMargins(0, 0, 0, 0);
		layout_.addWidget(&widget_);
	}

	{
		connect(&button_box_, &QDialogButtonBox::accepted, this, &SettingsDialog::OnOK);
		connect(&button_box_, &QDialogButtonBox::rejected, this, &QDialog::reject);
		layout_.addWidget(&button_box_);
	}
}
