#include "gui/widgets/anime_info.h"
#include "core/anime.h"
#include "core/anime_db.h"
#include "core/strings.h"
#include "gui/translate/anime.h"
#include "gui/widgets/text.h"
#include "services/services.h"
#include <QHBoxLayout>
#include <QTextStream>

AnimeInfoWidgetGetMetadataThread::AnimeInfoWidgetGetMetadataThread(QObject* parent) : QThread(parent) {}

void AnimeInfoWidgetGetMetadataThread::AddToQueue(int id) {
	const std::lock_guard<std::mutex> guard(queue_mutex_);
	queue_.push(id);
}

/* processes the queue... */
void AnimeInfoWidgetGetMetadataThread::run() {
	queue_mutex_.lock();
	while (!queue_.empty() && !isInterruptionRequested()) {
		int id = queue_.front();

		queue_mutex_.unlock();

		if (Services::RetrieveAnimeMetadata(id))
			emit NeedRefresh(id);

		queue_mutex_.lock();

		queue_.pop();
	}
	queue_mutex_.unlock();
}

/* all widgets share this thread */
static AnimeInfoWidgetGetMetadataThread get_metadata_thread;

AnimeInfoWidget::AnimeInfoWidget(QWidget* parent)
	: QWidget(parent)
	, _title(tr("Alternative titles"), "")
	, _details(tr("Details"), tr("Type:\nEpisodes:\nStatus:\nSeason:\nGenres:\nProducers:\nScore:"), "")
	, _synopsis(tr("Synopsis"), "") {
	QVBoxLayout* layout = new QVBoxLayout(this);

	layout->addWidget(&_title);
	layout->addWidget(&_details);
	layout->addWidget(&_synopsis);

	/* ... */
	connect(&get_metadata_thread, &AnimeInfoWidgetGetMetadataThread::NeedRefresh, this, [this](int id) {
		setUpdatesEnabled(false);

		if (id == id_)
			RefreshGenres(Anime::db.items[id]);

		setUpdatesEnabled(true);
	});
}

AnimeInfoWidget::AnimeInfoWidget(const Anime::Anime& anime, QWidget* parent) : AnimeInfoWidget(parent) {
	SetAnime(anime);
}

void AnimeInfoWidget::SetAnime(const Anime::Anime& anime) {
	setUpdatesEnabled(false);

	id_ = anime.GetId();

	get_metadata_thread.AddToQueue(id_);
	if (!get_metadata_thread.isRunning())
		get_metadata_thread.start();

	/* alt titles */
	_title.GetLine()->SetText(Strings::ToQString(Strings::Implode(anime.GetTitleSynonyms(), ", ")));

	RefreshGenres(anime);

	_synopsis.GetParagraph()->SetText(Strings::ToQString(anime.GetSynopsis()));

	setUpdatesEnabled(true);

	updateGeometry();
}

void AnimeInfoWidget::RefreshGenres(const Anime::Anime& anime) {
	/* details */
	QString details_data;
	QTextStream details_data_s(&details_data);

	/* we have to convert ALL of these strings to
	 * QString because QTextStream sucks and assumes
	 * Latin-1 (on Windows?) */
	const auto genres = anime.GetGenres();
	const auto producers = anime.GetProducers();

	details_data_s << Strings::ToQString(Translate::ToLocalString(anime.GetFormat())) << "\n"
	               << anime.GetEpisodes() << "\n"
	               << Strings::ToQString(Translate::ToLocalString(anime.GetAiringStatus())) << "\n"
	               << Strings::ToQString(Translate::ToLocalString(anime.GetSeason())) << " "
	               << anime.GetStartedDate().GetYear().value_or(2000) << "\n"
	               << Strings::ToQString((genres.size() > 1)    ? Strings::Implode(genres, ", ")    : "-") << "\n"
	               << Strings::ToQString((producers.size() > 1) ? Strings::Implode(producers, ", ") : "-") << "\n"
	               << anime.GetAudienceScore() << "%";

	_details.GetData()->setText(details_data);
}
