#ifndef __core__strings_h
#define __core__strings_h

#include <string>
#include <vector>
#include <set>
#include <sstream>

#include <cstdint>

class QString;
class QByteArray;

namespace Strings {

/* Implode function: takes a vector of strings and turns it
 * into a string, separated by delimiters.
*/
std::string Implode(const std::vector<std::string>& vector, const std::string& delimiter);
std::string Implode(const std::set<std::string>& set, const std::string& delimiter);
std::vector<std::string> Split(const std::string &text, const std::string& delimiter);

/* Substring removal functions */
std::string ReplaceAll(std::string string, const std::string& find, const std::string& replace);
std::string SanitizeLineEndings(const std::string& string);
std::string RemoveHtmlTags(std::string string);
std::string ParseHtmlEntities(std::string string);

/* stupid HTML bullshit */
std::string TextifySynopsis(const std::string& string);

std::string ToUpper(const std::string& string);
std::string ToLower(const std::string& string);

/* functions that make the way we convert from and to
 * different string formats universal (and these functions
 * typically do things the right way so we avoid retarded
 * code)
*/
std::wstring ToWstring(const std::string& string);
std::wstring ToWstring(const QString& string);
std::string ToUtf8String(const std::wstring& wstring);
std::string ToUtf8String(const QString& string);
std::string ToUtf8String(const QByteArray& ba);
QString ToQString(const std::string& string);
QString ToQString(const std::wstring& wstring);

/* not really an "int"... but who cares? */
template<typename T = int,
         std::enable_if_t<std::is_integral<T>::value, bool> = true>
T ToInt(const std::string& str, T def = 0) {
	std::istringstream s(str);
	s >> std::noboolalpha >> def;
	return def;
}

template<typename T,
         std::enable_if_t<std::is_integral<T>::value && !std::is_same<T, bool>::value, bool> = true>
std::string ToUtf8String(T i) {
	std::ostringstream s;
	s << i;
	return s.str();
}

bool ToBool(const std::string& s, bool def);

std::string ToUtf8String(bool b);

uint64_t HumanReadableSizeToBytes(const std::string& str);

std::string RemoveLeadingChars(std::string s, const char c);
std::string RemoveTrailingChars(std::string s, const char c);

bool BeginningMatchesSubstring(const std::string& str, const std::string& sub);

}; // namespace Strings

#endif // __core__strings_h
