#include "gui/widgets/anime_button.h"

#include "core/anime_db.h"
#include "core/session.h"
#include "core/strings.h"
#include "gui/widgets/elided_label.h"
#include "gui/widgets/poster.h"
#include "gui/widgets/text.h"

#include <QDate>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QWidget>

#include <iostream>

/* This widget is only used on the Seasons page. */

/***********************************\
*|---------| Title                  *
*|         |                        *
*|         | Aired                  *
*|         | Episodes               *
*| Poster  | Producers              *
*|         | Score                  *
*|         | Popularity             *
*|         |                        *
*|_________| Synopsis               *
\***********************************/

AnimeButton::AnimeButton(QWidget* parent) : QFrame(parent) {
	setFrameShadow(QFrame::Plain);
	setFrameShape(QFrame::Box);

	QHBoxLayout* ly = new QHBoxLayout(this);

	_poster.SetClickable(false);
	ly->addWidget(&_poster, 1, Qt::AlignTop);

	const std::vector<std::pair<std::string, std::string>> imap = {
		{Strings::Translate("Aired:"), ""},
		{Strings::Translate("Episodes:"), ""},
		{Strings::Translate("Genres:"), ""},
		{Strings::Translate("Producers:"), ""},
		{Strings::Translate("Score:"), ""},
		{Strings::Translate("Popularity:"), ""},
	};

	_info.SetData(imap);

	{
		QWidget* misc_section = new QWidget(this);

		QVBoxLayout* misc_layout = new QVBoxLayout(misc_section);
		misc_layout->setContentsMargins(0, 0, 0, 0);

		_title.setAutoFillBackground(true);
		_title.setContentsMargins(4, 4, 4, 4);
		_title.setStyleSheet("background-color: rgba(0, 245, 25, 50);");
		{
			QFont fnt(_title.font());
			fnt.setWeight(QFont::Bold);
			_title.setFont(fnt);
		}
		misc_layout->addWidget(&_title);

		_info.SetStyle(TextWidgets::LabelledParagraph::Style::BoldedLabels);

		_info.setContentsMargins(4, 0, 4, 0);
		_info.setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Minimum);
		misc_layout->addWidget(&_info);

		{
			QWidget* dummy = new QWidget(misc_section);
			dummy->setContentsMargins(4, 0, 4, 0);
			QVBoxLayout* dummy_layout = new QVBoxLayout(dummy);
			dummy_layout->setSpacing(0);
			dummy_layout->setContentsMargins(0, 0, 0, 0);

			_synopsis.SetSelectable(false);
			dummy_layout->addWidget(&_synopsis);
			misc_layout->addWidget(dummy);
		}

		ly->addWidget(misc_section, 3, Qt::AlignTop);
	}
}

AnimeButton::AnimeButton(const Anime::Anime& anime, QWidget* parent) : AnimeButton(parent) {
	SetAnime(anime);
}

void AnimeButton::SetAnime(const Anime::Anime& anime) {
	_poster.SetAnime(anime);
	_title.setText(Strings::ToQString(anime.GetUserPreferredTitle()));

	const QLocale& locale = session.config.locale.GetLocale();

	const std::vector<std::pair<std::string, std::string>> imap = {
		{Strings::Translate("Aired:"), Strings::ToUtf8String(locale.toString(anime.GetStartedDate().GetAsQDate(), "dd MMM yyyy"))},
		{Strings::Translate("Episodes:"), Strings::ToUtf8String(anime.GetEpisodes())},
		{Strings::Translate("Genres:"), Strings::Implode(anime.GetGenres(), ", ")},
		{Strings::Translate("Producers:"), "..."},
		{Strings::Translate("Score:"), Strings::ToUtf8String(anime.GetAudienceScore()) + "%"},
		{Strings::Translate("Popularity:"), "..."},
	};

	_info.SetData(imap);

	_synopsis.SetText(anime.GetSynopsis());
}

bool AnimeButton::hasHeightForWidth() const {
	return true;
}

int AnimeButton::heightForWidth(int w) const {
	return static_cast<int>(static_cast<double>(w) / 2.5);
}
