#ifndef MINORI_CORE_STRINGS_H_
#define MINORI_CORE_STRINGS_H_

#include <set>
#include <sstream>
#include <string>
#include <vector>

#include <cstdint>

class QString;
class QByteArray;

namespace Strings {

/* Implode function: takes a vector of strings and turns it
 * into a string, separated by delimiters.
 */
std::string Implode(const std::vector<std::string>& vector, const std::string& delimiter);
std::vector<std::string> Split(const std::string& text, const std::string& delimiter);

/* Substring removal functions */
void ReplaceAll(std::string& string, std::string_view find, std::string_view replace);
void SanitizeLineEndings(std::string& string);
void RemoveHtmlTags(std::string& string);
void ParseHtmlEntities(std::string& string);
void NormalizeUnicode(std::string& string);
void NormalizeAnimeTitle(std::string& string);

/* stupid HTML bullshit */
void TextifySynopsis(std::string& string);

std::string ToUpper(const std::string& string);
std::string ToLower(const std::string& string);

/* functions that make the way we convert from and to
 * different string formats universal (and these functions
 * typically do things the right way so we avoid retarded
 * code)
 */
std::wstring ToWstring(const std::string& string);
std::wstring ToWstring(const QString& string);
std::string ToUtf8String(const std::wstring& wstring);
std::string ToUtf8String(const QString& string);
std::string ToUtf8String(const QByteArray& ba);
QString ToQString(const std::string& string);
QString ToQString(const std::wstring& wstring);

/* not really an "int"... but who cares? */
template<typename T = int, std::enable_if_t<std::is_arithmetic<T>::value, bool> = true>
T ToInt(const std::string& str, T def = 0) {
	std::istringstream s(str);
	s >> std::noboolalpha >> def;
	return def;
}

template<typename T, std::enable_if_t<std::is_integral<T>::value && !std::is_same<T, bool>::value, bool> = true>
std::string ToUtf8String(T i) {
	std::ostringstream s;
	s << i;
	return s.str();
}

bool ToBool(const std::string& s, bool def);

std::string ToUtf8String(bool b);

uint64_t HumanReadableSizeToBytes(const std::string& str);
std::string BytesToHumanReadableSize(uint64_t bytes, int precision = 2);

void RemoveLeadingChars(std::string& s,  const char c);
void RemoveTrailingChars(std::string& s, const char c);

bool BeginningMatchesSubstring(const std::string& str, const std::string& sub);

}; // namespace Strings

#endif // MINORI_CORE_STRINGS_H_
