#include "gui/dialog/licenses.h"
#include "core/json.h"
#include "core/session.h"
#include "core/strings.h"
#include "gui/widgets/text.h"

#include <fmt/core.h>

#include <QCoreApplication>
#include <QFont>
#include <QHBoxLayout>
#include <QTabWidget>
#include <QTextBrowser>
#include <QTextCharFormat>
#include <QTextCursor>
#include <QFile>

#include <curl/curl.h>
#ifdef WIN32
#	include "sys/win32/dark_theme.h"
#endif

static QWidget *create_license_widget(QWidget *parent, const QString& license) {
	QTextBrowser* paragraph = new QTextBrowser(parent);
	paragraph->setFrameShape(QFrame::NoFrame);
	paragraph->setPlainText(license);
	paragraph->setFont(QFont("monospace"));
	return paragraph;
}

static void create_basic_license(QTabWidget *tab_widget, const QString& filename, const QString& title) {
	QFile f(filename);
	if (!f.exists())
		return;

	f.open(QFile::ReadOnly | QFile::Text);
	tab_widget->addTab(create_license_widget(tab_widget, f.readAll()), title);
}

static void create_dual_license(QTabWidget *tab_widget, const QString& filename1, const QString& title1, const QString& filename2, const QString& title2) {
	QString l1, l2;
	{
		QFile f1(filename1), f2(filename2);
		if (!f1.exists() || !f2.exists())
			return;

		f1.open(QFile::ReadOnly | QFile::Text);
		f2.open(QFile::ReadOnly | QFile::Text);

		l1 = f1.readAll();
		l2 = f2.readAll();
	}

	QWidget *dual = new QWidget(tab_widget);
	QVBoxLayout *dual_layout = new QVBoxLayout(dual);

	QLabel *dual_notice = new QLabel(QCoreApplication::tr("%1 was originally forked from %2, where any changes divergent from %2 are now under a different license. Both the licenses for %1 and %2 are provided below, respectfully:").arg(title1, title2), dual);
	dual_notice->setWordWrap(true);
	dual_layout->addWidget(dual_notice);
	dual_layout->addWidget(create_license_widget(dual, l1));
	dual_layout->addWidget(create_license_widget(dual, l2));

	tab_widget->addTab(dual, title1);
}

LicensesWindow::LicensesWindow(QWidget* parent) : QDialog(parent) {
	resize(641, 500);
	setWindowTitle(tr("About Minori"));
	setWindowFlags(Qt::Dialog | Qt::CustomizeWindowHint | Qt::WindowTitleHint | Qt::WindowCloseButtonHint);
	setAutoFillBackground(true);

	QHBoxLayout* layout = new QHBoxLayout(this);

	setBackgroundRole(QPalette::Base);

	QTabWidget *tab_widget = new QTabWidget(this);

	layout->addWidget(tab_widget);

	create_basic_license(tab_widget, ":/licenses/LICENSE.minori", tr("Minori"));

	create_dual_license(tab_widget, ":/licenses/LICENSE.MIT.animone", tr("Animone"), ":/licenses/LICENSE.BSD.animone", tr("Anisthesia"));

	create_basic_license(tab_widget, ":/licenses/LICENSE.anitomy", tr("Anitomy"));
	create_basic_license(tab_widget, ":/licenses/LICENSE.fmt", tr("fmt"));
	create_basic_license(tab_widget, ":/licenses/LICENSE.nlohmann", tr("JSON for Modern C++"));
	create_basic_license(tab_widget, ":/licenses/LICENSE.semver", tr("semver"));
	create_basic_license(tab_widget, ":/licenses/LICENSE.toml11", tr("toml11"));
	create_basic_license(tab_widget, ":/licenses/LICENSE.utf8proc", tr("utf8proc"));
}

void LicensesWindow::showEvent(QShowEvent* event) {
	QDialog::showEvent(event);
#ifdef WIN32
	win32::SetTitleBarsToBlack(session.config.theme.IsInDarkTheme());
#endif
}
