/*
** fd/xnu.cpp
**  - provides support for XNU (part of Darwin)
*/
#include "animia/fd/xnu.h"
#include "animia.h"

#include <cassert>
#include <string>
#include <unordered_map>
#include <vector>
#include <memory>

#include <fcntl.h>
#include <libproc.h>
#include <sys/sysctl.h>
#include <sys/types.h>
#include <sys/user.h>

#include <iostream>

namespace animia::internal::xnu {

static bool GetProcessNameFromArgs(pid_t pid, std::string& result) {
	/* sysctl shouldn't touch these, so we define them as const */
	const int mib[3] = {CTL_KERN, KERN_PROCARGS2, static_cast<int>(pid)};
	const size_t mib_size = sizeof(mib)/sizeof(*mib);

	/* Get the initial size of the array */
	size_t size;
	int ret = sysctl((int*)mib, mib_size, nullptr, &size, nullptr, 0);
	if (ret)
		return false;

	/* Reserve the space for it in a std::string */
	std::string args;
	args.resize(size);

	/* Get the contents of argc and argv */
	ret = sysctl((int*)mib, mib_size, &args.front(), &size, NULL, 0);
	if (ret)
		return false;

	/* Is the size big enough to hold at least argc? */
	if (size < sizeof(int))
		return false;

	args.resize(size);

	/* Get argc using memcpy */
	int argc;
	memcpy(&argc, &args.front(), sizeof(argc));

	/* Check for a condition that, realistically, would never happen,
	   but better to be safe than sorry */
	if (argc < 1)
		return false;

	/* Find the first null character */
	size_t null_pos = args.find('\0', sizeof(argc));
	if (null_pos == std::string::npos)
		return false;

	/* Find the last slash */
	size_t last_slash = args.rfind('/', null_pos);
	if (last_slash == std::string::npos)
		return false;

	/* Return our result */
	result = args.substr(last_slash + 1, null_pos - last_slash - 1);
	return true;
}

static bool GetProcessNameFromKernel(pid_t pid, std::string& result) {
	result.reserve(2*MAXCOMLEN);
	if (!proc_name(pid, &result.front(), result.length()))
		return false;

	result.shrink_to_fit();
	return true;
}

static bool GetProcessName(pid_t pid, std::string& result) {
	/* First try parsing the arguments, this prevents the process name being
	   cut off to 2*MAXCOMLEN (32 chars) */
	if (GetProcessNameFromArgs(pid, result))
		return true;

	/* Then attempt getting it from the kernel, which results in the process name
	   being cut to 32 chars (16 chars if p_name is unavailable) */
	if (GetProcessNameFromKernel(pid, result))
		return true;

	return false;
}

bool XnuFdTools::EnumerateOpenProcesses(process_proc_t process_proc) {
	size_t pids_size = 512;
	std::unique_ptr<pid_t[]> pids;

	int returned_size = 0;
	do {
		pids.reset(new pid_t[pids_size]);
		returned_size = proc_listpids(PROC_ALL_PIDS, 0, pids.get(), pids_size * sizeof(pid_t));
		if (returned_size == -1)
			return false;
	} while ((pids_size * sizeof(size_t)) < returned_size);

	for (int i = 0; i < pids_size; i++) {
		std::string result;
		GetProcessName(pids[i], result);
		if (!process_proc({pids[i], result}))
			return false;
	}

	return true;
}

bool XnuFdTools::EnumerateOpenFiles(const std::set<pid_t>& pids, open_file_proc_t open_file_proc) {
	if (!open_file_proc)
		return false;

	for (const auto& pid : pids) {
		int bufsz = proc_pidinfo(pid, PROC_PIDLISTFDS, 0, NULL, 0);
		if (bufsz == -1)
			return false;

		struct proc_fdinfo* info = (struct proc_fdinfo*)malloc(bufsz);
		if (!info)
			return false;

		proc_pidinfo(pid, PROC_PIDLISTFDS, 0, info, bufsz);

		for (int i = 0; i < bufsz / sizeof(info[0]); i++) {
			if (info[i].proc_fdtype == PROX_FDTYPE_VNODE) {
				struct vnode_fdinfowithpath vnodeInfo;

				int sz = proc_pidfdinfo(pid, info[i].proc_fd, PROC_PIDFDVNODEPATHINFO, &vnodeInfo,
				                        PROC_PIDFDVNODEPATHINFO_SIZE);
				if (sz != PROC_PIDFDVNODEPATHINFO_SIZE)
					return false;

				/* this doesn't work!
				if (vnodeInfo.pfi.fi_openflags & O_WRONLY || vnodeInfo.pfi.fi_openflags & O_RDWR)
				    continue;
				*/

				if (!open_file_proc({pid, vnodeInfo.pvip.vip_path}))
					return false;
			}
		}
	}

	return true;
}

} // namespace animia::internal::xnu
