#include "animone/fd/xnu.h"
#include "animone.h"
#include "animone/util/osx.h"

#include <cassert>
#include <memory>
#include <string>
#include <unordered_map>
#include <vector>

#include <fcntl.h>
#include <libproc.h>
#include <sys/sysctl.h>
#include <sys/types.h>
#include <sys/user.h>

namespace animone::internal::xnu {

bool EnumerateOpenProcesses(process_proc_t process_proc) {
	size_t pids_size = 256;
	std::unique_ptr<pid_t[]> pids;

	int returned_size = 0;
	do {
		pids.reset(new pid_t[pids_size *= 2]);
		returned_size = proc_listpids(PROC_ALL_PIDS, 0, pids.get(), pids_size * sizeof(pid_t));
		if (returned_size == -1)
			return false;
	} while ((pids_size * sizeof(size_t)) < returned_size);

	for (int i = 0; i < pids_size; i++) {
		std::string result;
		osx::util::GetProcessName(pids[i], result);
		if (!process_proc({pids[i], result}))
			return false;
	}

	return true;
}

bool EnumerateOpenFiles(const std::set<pid_t>& pids, open_file_proc_t open_file_proc) {
	if (!open_file_proc)
		return false;

	for (const auto& pid : pids) {
		const int bufsz = proc_pidinfo(pid, PROC_PIDLISTFDS, 0, NULL, 0);
		if (bufsz < 0)
			return false;

		const size_t info_len = bufsz / sizeof(struct proc_fdinfo);
		if (info_len < 1)
			return false;

		std::unique_ptr<struct proc_fdinfo[]> info(new struct proc_fdinfo[info_len]);
		if (!info)
			return false;

		proc_pidinfo(pid, PROC_PIDLISTFDS, 0, info.get(), bufsz);

		for (size_t i = 0; i < info_len; i++) {
			if (info[i].proc_fdtype == PROX_FDTYPE_VNODE) {
				struct vnode_fdinfowithpath vnodeInfo;

				int sz = proc_pidfdinfo(pid, info[i].proc_fd, PROC_PIDFDVNODEPATHINFO, &vnodeInfo,
				                        PROC_PIDFDVNODEPATHINFO_SIZE);
				if (sz != PROC_PIDFDVNODEPATHINFO_SIZE)
					return false;

				// This doesn't work (for unknown reasons). I assume somethings fucked up with
				// my assumptions; I don't care enough to look into it tbh
				//
				// if (vnodeInfo.pfi.fi_openflags & O_WRONLY || vnodeInfo.pfi.fi_openflags & O_RDWR)
				//     continue;

				if (!open_file_proc({pid, vnodeInfo.pvip.vip_path}))
					return false;
			}
		}
	}

	return true;
}

} // namespace animone::internal::xnu
