#include "gui/widgets/text.h"
#include "core/session.h"

#include <QDebug>
#include <QFrame>
#include <QLabel>
#include <QTextBlock>
#include <QVBoxLayout>

namespace TextWidgets {

Header::Header(const QString& title, QWidget* parent) : QWidget(parent) {
	QVBoxLayout* layout = new QVBoxLayout(this);
	setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum);

	static_text_title = new QLabel(title, this);
	static_text_title->setTextFormat(Qt::PlainText);

	{
		QFont font = static_text_title->font();
		font.setWeight(QFont::Bold);
		static_text_title->setFont(font);
	}

	static_text_line = new QFrame(this);
	static_text_line->setFrameShape(QFrame::HLine);
	static_text_line->setFrameShadow(QFrame::Sunken);
	static_text_line->setFixedHeight(2);

	layout->addWidget(static_text_title);
	layout->addWidget(static_text_line);
	layout->setSpacing(0);
	layout->setContentsMargins(0, 0, 0, 0);
}

void Header::SetText(const QString& text) {
	static_text_title->setText(text);
	updateGeometry();
}

/* for now, this is a QLabel with a couple of default settings.
 *
 * eventually I'll have to implement this as a QScrollArea, just in case
 * some random text decides to overflow or something.
 */
Paragraph::Paragraph(const QString& text, QWidget* parent) : QLabel(text, parent) {
	setTextInteractionFlags(Qt::TextBrowserInteraction);
	setFrameShape(QFrame::NoFrame);
	setCursor(Qt::IBeamCursor); /* emulate Taiga */
	setWordWrap(true);

	setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Minimum);
}

/* kept here for legacy reasons, see explanation above */
void Paragraph::SetText(const QString& text) {
	setText(text);
}

/* Equivalent to Paragraph(), but disables word wrap. */
Line::Line(QWidget* parent) : Paragraph("", parent) {
	setWordWrap(false);
}

Line::Line(const QString& text, QWidget* parent) : Paragraph(text, parent) {
	setWordWrap(false);
}

/* legacy function, don't use in new code */
void Line::SetText(const QString& text) {
	setText(text);
}

Title::Title(const QString& title, QWidget* parent) : Line(title, parent) {
	QFont fnt(font());
	fnt.setPixelSize(16);
	setFont(fnt);

	QPalette pal(palette());
	pal.setColor(QPalette::Text, pal.color(QPalette::Highlight));
	setPalette(pal);
}

Section::Section(const QString& title, const QString& data, QWidget* parent) : QWidget(parent) {
	QVBoxLayout* layout = new QVBoxLayout(this);

	header = new Header(title, this);

	QWidget* content = new QWidget(this);
	QHBoxLayout* content_layout = new QHBoxLayout(content);

	paragraph = new Paragraph(data, this);
	paragraph->setTextInteractionFlags(Qt::NoTextInteraction);
	paragraph->setAttribute(Qt::WidgetAttribute::WA_TransparentForMouseEvents);
	paragraph->setWordWrap(QTextOption::NoWrap);

	content_layout->addWidget(paragraph);
	content_layout->setSpacing(0);
	content_layout->setContentsMargins(0, 0, 0, 0);
	content->setContentsMargins(12, 0, 0, 0);

	layout->addWidget(header);
	layout->addWidget(paragraph);
	layout->setSpacing(0);
	layout->setContentsMargins(0, 0, 0, 0);
}

Header* Section::GetHeader() {
	return header;
}

Paragraph* Section::GetParagraph() {
	return paragraph;
}

LabelledParagraph::LabelledParagraph(const QString& label, const QString& data, QWidget* parent) : QWidget(parent) {
	QHBoxLayout* ly = new QHBoxLayout(this);

	labels = new Paragraph(label, this);
	labels->setTextInteractionFlags(Qt::NoTextInteraction);
	labels->setAttribute(Qt::WidgetAttribute::WA_TransparentForMouseEvents);
	labels->setWordWrap(QTextOption::NoWrap);
	labels->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Expanding);

	paragraph = new Paragraph(data, this);
	paragraph->setTextInteractionFlags(Qt::NoTextInteraction);
	paragraph->setAttribute(Qt::WidgetAttribute::WA_TransparentForMouseEvents);
	paragraph->setWordWrap(QTextOption::NoWrap);
	paragraph->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding);

	ly->addWidget(labels, 0, Qt::AlignTop);
	ly->addWidget(paragraph, 0, Qt::AlignTop);
	ly->setSpacing(20);
	ly->setContentsMargins(0, 0, 0, 0);
}

Paragraph* LabelledParagraph::GetLabels() {
	return labels;
}

Paragraph* LabelledParagraph::GetParagraph() {
	return paragraph;
}

LabelledSection::LabelledSection(const QString& title, const QString& label, const QString& data, QWidget* parent)
    : QWidget(parent) {
	QVBoxLayout* layout = new QVBoxLayout(this);

	header = new Header(title, this);

	// this is not accessible from the object because there's really
	// no reason to make it accessible...
	content = new LabelledParagraph(label, data, this);
	content->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Minimum);
	content->setContentsMargins(12, 0, 0, 0);

	layout->addWidget(header);
	layout->addWidget(content);
	layout->setSpacing(0);
	layout->setContentsMargins(0, 0, 0, 0);
}

Header* LabelledSection::GetHeader() {
	return header;
}

Paragraph* LabelledSection::GetLabels() {
	return content->GetLabels();
}

Paragraph* LabelledSection::GetParagraph() {
	return content->GetParagraph();
}

SelectableSection::SelectableSection(const QString& title, const QString& data, QWidget* parent) : QWidget(parent) {
	QVBoxLayout* layout = new QVBoxLayout(this);

	header = new Header(title, this);

	QWidget* content = new QWidget(this);
	QHBoxLayout* content_layout = new QHBoxLayout(content);

	paragraph = new Paragraph(data, content);

	content_layout->addWidget(paragraph);
	content_layout->setSpacing(0);
	content_layout->setContentsMargins(12, 0, 0, 0);
	content->setContentsMargins(0, 0, 0, 0);

	layout->addWidget(header);
	layout->addWidget(content);
	layout->setSpacing(0);
	layout->setContentsMargins(0, 0, 0, 0);
}

Header* SelectableSection::GetHeader() {
	return header;
}

Paragraph* SelectableSection::GetParagraph() {
	return paragraph;
}

OneLineSection::OneLineSection(const QString& title, const QString& text, QWidget* parent) : QWidget(parent) {
	QVBoxLayout* layout = new QVBoxLayout(this);

	header = new Header(title, this);

	QWidget* content = new QWidget(this);
	QHBoxLayout* content_layout = new QHBoxLayout(content);

	line = new Line(text, content);

	content_layout->addWidget(line);
	content_layout->setSpacing(0);
	content_layout->setContentsMargins(0, 0, 0, 0);
	content->setContentsMargins(12, 0, 0, 0);

	layout->addWidget(header);
	layout->addWidget(content);
	layout->setSpacing(0);
	layout->setContentsMargins(0, 0, 0, 0);
}

Header* OneLineSection::GetHeader() {
	return header;
}

Line* OneLineSection::GetLine() {
	return line;
}

} // namespace TextWidgets
