/**
 * config.cpp:
 * parses the config... lol
 **/
#include "core/config.h"
#include "core/anime.h"
#include "core/filesystem.h"
#include "core/ini.h"
#include "core/json.h"
#include "core/strings.h"
#include "gui/translate/anime.h"
#include "gui/translate/config.h"

#include "animone/player.h"

#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <filesystem>
#include <fstream>
#include <limits.h>

#include <QFile>
#include <QTextStream>

#include <iostream>

/* I'll use an INI-based config file instead of using an
 * XML file like Taiga.
 *
 * It technically isn't to spec, because I'm making these case-sensitive.
 * Boohoo.
 */

int Config::Load() {
	std::filesystem::path cfg_path = Filesystem::GetConfigPath();

	mINI::INIFile file(cfg_path.u8string());
	mINI::INIStructure ini;
	file.read(ini);

	service = Translate::ToService(INI::GetIniString(ini, "General", "Service", "None"));

	anime_list.score_format =
	    Translate::ToScoreFormat(INI::GetIniString(ini, "Anime List", "Score format", "POINT_100"));
	anime_list.language =
	    Translate::ToLanguage(INI::GetIniString(ini, "Anime List", "Title language", "Romaji"));
	anime_list.display_aired_episodes = INI::GetIniBool(ini, "Anime List", "Display only aired episodes", true);
	anime_list.display_available_episodes =
	    INI::GetIniBool(ini, "Anime List", "Display only available episodes in library", true);
	anime_list.highlight_anime_if_available =
	    INI::GetIniBool(ini, "Anime List", "Highlight anime if available", true);

	if (anime_list.highlight_anime_if_available) // sanity check
		anime_list.highlighted_anime_above_others =
		    INI::GetIniBool(ini, "Anime List", "Display highlighted anime above others", false);
	else
		anime_list.highlighted_anime_above_others = false;

	auth.anilist.auth_token = INI::GetIniString(ini, "Authentication/AniList", "Auth Token", "");
	auth.anilist.user_id = INI::GetIniInteger<int>(ini, "Authentication/AniList", "User ID", 0);

	auth.kitsu.access_token = INI::GetIniString(ini, "Authentication/Kitsu", "Access Token", "");
	auth.kitsu.access_token_expiration = INI::GetIniInteger<Time::Timestamp>(ini, "Authentication/Kitsu", "Access Token Expiration", 0);
	auth.kitsu.refresh_token = INI::GetIniString(ini, "Authentication/Kitsu", "Refresh Token", "");
	auth.kitsu.user_id = INI::GetIniString(ini, "Authentication/Kitsu", "User ID", "");

	torrents.feed_link = INI::GetIniString(ini, "Torrents", "RSS feed",
	                                       "https://www.tokyotosho.info/rss.php?filter=1,11&zwnj=0");

	recognition.detect_media_players = INI::GetIniBool(ini, "Recognition", "Detect media players", true);

	/* lots of dumb logic to import the player data */
	{
		/* load the player data */
		QFile f(":/players.anisthesia");
		if (!f.exists())
			return false;

		f.open(QFile::ReadOnly | QFile::Text);
		QTextStream ts(&f);

		std::vector<animone::Player> players;

		if (!animone::ParsePlayersData(Strings::ToUtf8String(ts.readAll()), players))
			return false;

		recognition.players.reserve(players.size());
		for (const auto& player : players)
			recognition.players.push_back({true, player});
	}

	for (auto& [enabled, player] : recognition.players) {
		switch (player.type) {
			default:
			case animone::PlayerType::Default:
				enabled = INI::GetIniBool(ini, "Recognition/Players", player.name, true);
				break;
			case animone::PlayerType::WebBrowser:
				enabled = INI::GetIniBool(ini, "Recognition/Browsers", player.name, true);
				break;
		}
	}

	locale.RefreshAvailableLocales();
	locale.SetActiveLocale(
	    QLocale(Strings::ToQString(INI::GetIniString(ini, "General", "Locale", "en_US"))));

	theme.SetTheme(Translate::ToTheme(INI::GetIniString(ini, "Appearance", "Theme", "Default")));

	{
		std::vector<std::string> v = Strings::Split(INI::GetIniString(ini, "Library", "Folders", ""), ";");
		for (const auto& s : v)
			if (!library.paths.count(s))
				library.paths.insert(s);
	}

	library.real_time_monitor = INI::GetIniBool(ini, "Library", "Real-time monitor", true);

	return 0;
}

int Config::Save() {
	std::filesystem::path cfg_path = Filesystem::GetConfigPath();
	Filesystem::CreateDirectories(cfg_path);

	mINI::INIFile file(cfg_path.string());
	mINI::INIStructure ini;

	ini["General"]["Service"] = Translate::ToString(service);
	ini["General"]["Locale"] = Strings::ToUtf8String(locale.GetLocale().name());

	ini["Anime List"]["Score format"] = Translate::ToString(anime_list.score_format);
	ini["Anime List"]["Title language"] = Translate::ToString(anime_list.language);
	ini["Anime List"]["Display only aired episodes"] = Strings::ToUtf8String(anime_list.display_aired_episodes);
	ini["Anime List"]["Display only available episodes in library"] = Strings::ToUtf8String(anime_list.display_available_episodes);
	ini["Anime List"]["Highlight anime if available"] = Strings::ToUtf8String(anime_list.highlight_anime_if_available);
	ini["Anime List"]["Display highlighted anime above others"] = Strings::ToUtf8String(anime_list.highlighted_anime_above_others);

	ini["Authentication/AniList"]["Auth Token"] = auth.anilist.auth_token;
	ini["Authentication/AniList"]["User ID"] = Strings::ToUtf8String(auth.anilist.user_id);

	ini["Authentication/Kitsu"]["Access Token"] = auth.kitsu.access_token;
	ini["Authentication/Kitsu"]["Access Token Expiration"] = Strings::ToUtf8String(auth.kitsu.access_token_expiration);
	ini["Authentication/Kitsu"]["Refresh Token"] = auth.kitsu.refresh_token;
	ini["Authentication/Kitsu"]["User ID"] = auth.kitsu.user_id;

	ini["Appearance"]["Theme"] = Translate::ToString(theme.GetTheme());

	ini["Torrents"]["RSS feed"] = torrents.feed_link;

	ini["Recognition"]["Detect media players"] = Strings::ToUtf8String(recognition.detect_media_players);

	for (const auto& [enabled, player] : recognition.players) {
		const std::string section = (player.type == animone::PlayerType::WebBrowser) ? "Recognition/Players" : "Recognition/Browsers";
		ini[section][player.name] = Strings::ToUtf8String(enabled);
	}

	ini["Library"]["Folders"] = Strings::Implode(library.paths, ";");
	ini["Library"]["Real-time monitor"] = Strings::ToUtf8String(library.real_time_monitor);

	file.write(ini);

	return 0;
}
