#include "gui/widgets/anime_button.h"

#include "core/anime_db.h"
#include "core/strings.h"
#include "core/session.h"
#include "gui/widgets/text.h"
#include "gui/widgets/elided_label.h"
#include "gui/widgets/poster.h"

#include <QWidget>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QDate>

/* This widget is only used on the Seasons page. */

/***********************************\
*|---------| Title                  *
*|         |                        *
*|         | Aired                  *
*|         | Episodes               *
*| Poster  | Producers              *
*|         | Score                  *
*|         | Popularity             *
*|         |                        *
*|_________| Synopsis               *
\***********************************/

AnimeButton::AnimeButton(QWidget* parent) : QFrame(parent) {
	setFrameShadow(QFrame::Plain);
	setFrameShape(QFrame::Box);
	QHBoxLayout* ly = new QHBoxLayout(this);

	_poster = new Poster(this);
	_poster->setFixedSize(120, 170);
	ly->addWidget(_poster);

	{
		QWidget* misc_section = new QWidget(this);
		misc_section->setFixedSize(354, 180);

		QVBoxLayout* misc_layout = new QVBoxLayout(misc_section);
		misc_layout->setContentsMargins(0, 0, 0, 0);

		_title = new TextWidgets::Line("", misc_section);
		misc_layout->addWidget(_title);

		/* need to make a separate "labelled paragraph" for this */
		_info = new TextWidgets::LabelledParagraph(tr("Aired:\nEpisodes:\nProducers:\nScore:\nPopularity:"), "\n\n\n\n", misc_section);
		_info->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Minimum);
		misc_layout->addWidget(_info);

		_synopsis = new ElidedLabel("", misc_section);
		misc_layout->addWidget(_synopsis);

		ly->addWidget(misc_section);
	}
}

AnimeButton::AnimeButton(const Anime::Anime& anime, QWidget* parent) : AnimeButton(parent) {
	SetAnime(anime);
}

void AnimeButton::SetAnime(const Anime::Anime& anime) {
	_poster->SetAnime(anime);
	_title->SetText(Strings::ToQString(anime.GetUserPreferredTitle()));

	{
		const QLocale& locale = session.config.locale.GetLocale();
		_info->GetParagraph()->SetText(
			locale.toString(anime.GetAirDate().GetAsQDate(), "dd MMM yyyy") + "\n" +
			QString::number(anime.GetEpisodes()) + "\n" +
			"...\n" +
			QString::number(anime.GetAudienceScore()) + "%\n" +
			"..."
		);
	}

	{
		QString synopsis = Strings::ToQString(anime.GetSynopsis());
		QFontMetrics metrics(_synopsis->font());
		_synopsis->SetText(Strings::ToQString(anime.GetSynopsis()));
	}
}