#include "gui/widgets/sidebar.h"
#include <QFrame>
#include <QListWidget>
#include <QListWidgetItem>
#include <QMouseEvent>

SideBar::SideBar(QWidget* parent) : QListWidget(parent) {
	setFrameShape(QFrame::NoFrame);
	setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	setSelectionMode(QAbstractItemView::SingleSelection);
	setSelectionBehavior(QAbstractItemView::SelectItems);
	setMouseTracking(true);

	setStyleSheet("QListWidget::item:disabled { background: transparent }");

	SetBackgroundTransparent(true);

	connect(this, &QListWidget::currentRowChanged, this,
	        [this](int index) { emit CurrentItemChanged(RemoveSeparatorsFromIndex(index)); });
}

void SideBar::SetCurrentItem(int index) {
	setCurrentRow(AddSeparatorsToIndex(index));
}

int SideBar::GetCurrentItem() {
	return RemoveSeparatorsFromIndex(currentRow());
}

void SideBar::SetBackgroundTransparent(bool yes) {
	viewport()->setAutoFillBackground(!yes);
}

QListWidgetItem* SideBar::AddItem(QString name, QIcon icon) {
	QListWidgetItem* item = new QListWidgetItem(this);
	item->setText(name);
	if (!icon.isNull())
		item->setIcon(icon);
	return item;
}

QIcon SideBar::CreateIcon(const char* file) {
	QPixmap pixmap(file, "PNG");
	QIcon result;
	result.addPixmap(pixmap, QIcon::Normal);
	result.addPixmap(pixmap, QIcon::Selected);
	return result;
}

QListWidgetItem* SideBar::AddSeparator() {
	QListWidgetItem* item = new QListWidgetItem(this);
	QFrame* line = new QFrame(this);
	line->setFrameShape(QFrame::HLine);
	line->setFrameShadow(QFrame::Sunken);
	line->setMouseTracking(true);
	line->setEnabled(false);

	setItemWidget(item, line);
	item->setFlags(Qt::NoItemFlags);
	item->setBackground(QBrush(Qt::transparent));
	return item;
}

int SideBar::AddSeparatorsToIndex(int index) {
	int separators = 0, items = 0;

	for (; items <= index;) {
		if (IndexIsSeparator(indexFromItem(item(items + separators)))) {
			separators++;
		} else {
			items++;
		}
	}

	return index + separators;
}

int SideBar::RemoveSeparatorsFromIndex(int index) {
	int i = 0, items = 0;
	for (; i < index; i++) {
		if (!IndexIsSeparator(indexFromItem(item(i))))
			items++;
	}
	return items;
}

bool SideBar::IndexIsSeparator(QModelIndex index) const {
	return !index.isValid() || !(index.flags() & Qt::ItemIsEnabled);
}

QItemSelectionModel::SelectionFlags SideBar::selectionCommand(const QModelIndex& index, const QEvent*) const {
	if (IndexIsSeparator(index))
		return QItemSelectionModel::NoUpdate;
	return QItemSelectionModel::ClearAndSelect;
}

void SideBar::mouseMoveEvent(QMouseEvent* event) {
	if (!IndexIsSeparator(indexAt(event->pos())))
		setCursor(Qt::PointingHandCursor);
	else
		unsetCursor();
	QListView::mouseMoveEvent(event);
}
