/*
** fd/xnu.cpp
**  - provides support for XNU (part of Darwin)
*/
#include "animia/fd/xnu.h"
#include "animia.h"

#include <cassert>
#include <string>
#include <unordered_map>
#include <vector>

#include <fcntl.h>
#include <libproc.h>
#include <sys/sysctl.h>
#include <sys/types.h>
#include <sys/user.h>

namespace animia::internal::xnu {

static std::string GetProcessName(pid_t pid) {
	struct proc_bsdinfo proc;

	int st = proc_pidinfo(pid, PROC_PIDTBSDINFO, 0, &proc, PROC_PIDTBSDINFO_SIZE);
	if (st != PROC_PIDTBSDINFO_SIZE)
		return "";

	return (proc.pbi_name[0]) ? proc.pbi_name : proc.pbi_comm;
}

bool XnuFdTools::EnumerateOpenProcesses(process_proc_t process_proc) {
	std::vector<pid_t> pids;
	pids.reserve(1024);

	for (int returned_size = pids.capacity(); pids.capacity() > returned_size; pids.reserve(pids.capacity() * 2)) {
		returned_size = proc_listpids(PROC_ALL_PIDS, 0, pids.data(), pids.capacity() * sizeof(pid_t));
		if (returned_size == -1)
			return false;
	}

	for (int i = 0; i < size / sizeof(*pids); i++) {
		const pid_t pid = pids[i].kp_proc.p_pid;
		if (!process_proc({pid, GetProcessName(pid)}))
			return false;
	}

	return true;
}

bool XnuFdTools::EnumerateOpenFiles(const std::set<pid_t>& pids, open_file_proc_t open_file_proc) {
	if (!open_file_proc)
		return false;

	for (const auto& pid : pids) {
		int bufsz = proc_pidinfo(pid, PROC_PIDLISTFDS, 0, NULL, 0);
		if (bufsz == -1)
			return false;

		struct proc_fdinfo* info = (struct proc_fdinfo*)malloc(bufsz);
		if (!info)
			return false;

		proc_pidinfo(pid, PROC_PIDLISTFDS, 0, info, bufsz);

		for (int i = 0; i < bufsz / sizeof(info[0]); i++) {
			if (info[i].proc_fdtype == PROX_FDTYPE_VNODE) {
				struct vnode_fdinfowithpath vnodeInfo;

				int sz = proc_pidfdinfo(pid, info[i].proc_fd, PROC_PIDFDVNODEPATHINFO, &vnodeInfo,
				                        PROC_PIDFDVNODEPATHINFO_SIZE);
				if (sz != PROC_PIDFDVNODEPATHINFO_SIZE)
					return false;

				/* this doesn't work!
				if (vnodeInfo.pfi.fi_openflags & O_WRONLY || vnodeInfo.pfi.fi_openflags & O_RDWR)
				    continue;
				*/

				if (!open_file_proc({pid, vnodeInfo.pvip.vip_path}))
					return false;
			}
		}
	}

	return true;
}

} // namespace animia::internal::xnu
