/*
* Copyright (C) 2016 The Qt Company Ltd.
* Contact: https://www.qt.io/licensing/
*
* This file is part of the QtCore module of the Qt Toolkit.
*
* "Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions are
* met:
*   * Redistributions of source code must retain the above copyright
*     notice, this list of conditions and the following disclaimer.
*   * Redistributions in binary form must reproduce the above copyright
*     notice, this list of conditions and the following disclaimer in
*     the documentation and/or other materials provided with the
*     distribution.
*   * Neither the name of The Qt Company Ltd nor the names of its
*     contributors may be used to endorse or promote products derived
*     from this software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
* A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
* OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
* SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
* THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
*/
#include "gui/layouts/flow_layout.h"

#include <QWidget>

#include <algorithm>

FlowLayout::FlowLayout(QWidget *parent, int margin, int hSpacing, int vSpacing)
	: QLayout(parent), _horiz_space(hSpacing), _vert_space(vSpacing) {
	setContentsMargins(margin, margin, margin, margin);
}

FlowLayout::FlowLayout(int margin, int hSpacing, int vSpacing)
	: _horiz_space(hSpacing), _vert_space(vSpacing) {
	setContentsMargins(margin, margin, margin, margin);
}

FlowLayout::~FlowLayout() {
	while (count())
		delete takeAt(0);
}

void FlowLayout::addItem(QLayoutItem *item) {
	item_list.append(item);
}

int FlowLayout::horizontalSpacing() const {
	return (_horiz_space >= 0) ? _horiz_space : smartSpacing(QStyle::PM_LayoutHorizontalSpacing);
}

int FlowLayout::verticalSpacing() const {
	return (_vert_space >= 0) ? _vert_space : smartSpacing(QStyle::PM_LayoutVerticalSpacing);
}

int FlowLayout::count() const {
	return item_list.size();
}

QLayoutItem* FlowLayout::itemAt(int index) const {
	return item_list.value(index);
}

QLayoutItem* FlowLayout::takeAt(int index) {
	return (index >= 0 && index < item_list.size()) ? item_list.takeAt(index) : nullptr;
}

Qt::Orientations FlowLayout::expandingDirections() const
{
	return {};
}

bool FlowLayout::hasHeightForWidth() const {
	return true;
}

int FlowLayout::heightForWidth(int width) const {
	return doLayout(QRect(0, 0, width, 0), true);
}

void FlowLayout::setGeometry(const QRect &rect) {
	QLayout::setGeometry(rect);
	doLayout(rect, false);
}

QSize FlowLayout::sizeHint() const {
	return minimumSize();
}

QSize FlowLayout::minimumSize() const {
	QSize size;
	for (QLayoutItem* item : item_list)
		size = size.expandedTo(item->minimumSize());

	const QMargins margins = contentsMargins();
	size += QSize(margins.left() + margins.right(), margins.top() + margins.bottom());
	return size;
}

int FlowLayout::doLayout(const QRect &rect, bool test) const {
	int left, top, right, bottom;
	getContentsMargins(&left, &top, &right, &bottom);
	QRect effectiveRect = rect.adjusted(+left, +top, -right, -bottom);

	int x = effectiveRect.x();
	int y = effectiveRect.y();
	int line_height = 0;

	for (QLayoutItem* item : item_list) {
		const QWidget* wid = item->widget();
		int horiz_space = horizontalSpacing();
		if (horiz_space == -1)
			horiz_space = wid->style()->layoutSpacing(QSizePolicy::PushButton, QSizePolicy::PushButton, Qt::Horizontal);

		int vert_space = verticalSpacing();
		if (vert_space == -1)
			vert_space = wid->style()->layoutSpacing(QSizePolicy::PushButton, QSizePolicy::PushButton, Qt::Vertical);

		int next_x = x + item->sizeHint().width() + horiz_space;
		if ((next_x - horiz_space > effectiveRect.right()) && (line_height > 0)) {
			x = effectiveRect.x();
			y = y + line_height + vert_space;
			next_x = x + item->sizeHint().width() + horiz_space;
			line_height = 0;
		}

		if (!test)
			item->setGeometry(QRect(QPoint(x, y), item->sizeHint()));

		x = next_x;
		line_height = std::max(line_height, item->sizeHint().height());
	}

	return y + line_height - rect.y() + bottom;
}

int FlowLayout::smartSpacing(QStyle::PixelMetric pm) const {
	QObject *parent = this->parent();
	if (!parent) {
		return -1;
	} else if (parent->isWidgetType()) {
		QWidget *pw = static_cast<QWidget *>(parent);
		return pw->style()->pixelMetric(pm, nullptr, pw);
	} else {
		return static_cast<QLayout *>(parent)->spacing();
	}
}
