#include "core/anime_db.h"
#include "core/anime.h"
#include "core/strings.h"
#include "core/json.h"

#include <QDebug>

namespace Anime {

int Database::GetTotalAnimeAmount() {
	int total = 0;
	for (const auto& a : items) {
		if (a.second.IsInUserList())
			total++;
	}
	return total;
}

int Database::GetListsAnimeAmount(ListStatus status) {
	if (status == ListStatus::NOT_IN_LIST)
		return 0;
	int total = 0;
	for (const auto& a : items) {
		if (a.second.IsInUserList() && a.second.GetUserStatus() == status)
			total++;
	}
	return total;
}

int Database::GetTotalEpisodeAmount() {
	int total = 0;
	for (const auto& a : items) {
		if (a.second.IsInUserList()) {
			total += a.second.GetUserRewatchedTimes() * a.second.GetEpisodes();
			total += a.second.GetUserProgress();
		}
	}
	return total;
}

/* Returns the total watched amount in minutes. */
int Database::GetTotalWatchedAmount() {
	int total = 0;
	for (const auto& a : items) {
		if (a.second.IsInUserList()) {
			total += a.second.GetDuration() * a.second.GetUserProgress();
			total += a.second.GetEpisodes() * a.second.GetDuration() * a.second.GetUserRewatchedTimes();
		}
	}
	return total;
}

/* Returns the total planned amount in minutes.
   Note that we should probably limit progress to the
   amount of episodes, as AniList will let you
   set episode counts up to 32768. But that should
   rather be handled elsewhere. */
int Database::GetTotalPlannedAmount() {
	int total = 0;
	for (const auto& a : items) {
		if (a.second.IsInUserList())
			total += a.second.GetDuration() * (a.second.GetEpisodes() - a.second.GetUserProgress());
	}
	return total;
}

/* In Taiga this is called a mean, but "average" is
   what's primarily used in conversation, at least
   in the U.S. */
double Database::GetAverageScore() {
	double avg = 0;
	int amt = 0;
	for (const auto& a : items) {
		if (a.second.IsInUserList() && a.second.GetUserScore()) {
			avg += a.second.GetUserScore();
			amt++;
		}
	}
	return avg / amt;
}

double Database::GetScoreDeviation() {
	double squares_sum = 0, avg = GetAverageScore();
	int amt = 0;
	for (const auto& a : items) {
		if (a.second.IsInUserList() && a.second.GetUserScore()) {
			squares_sum += std::pow(static_cast<double>(a.second.GetUserScore()) - avg, 2);
			amt++;
		}
	}
	return (amt > 0) ? std::sqrt(squares_sum / amt) : 0;
}

template <typename T, typename U>
static T get_lowest_in_map(const std::unordered_map<T, U>& map) {
	if (map.size() <= 0)
		return 0;

	T id = 0;
	U ret = std::numeric_limits<U>::max();
	for (const auto& t : map) {
		if (t.second < ret) {
			ret = t.second;
			id = t.first;
		}
	}
	return id;
}

/* This is really fugly but WHO CARES :P

   This fairly basic algorithm is only in effect because
   there are some special cases, e.g. Another and Re:ZERO, where 
   we get the wrong match, so we have to create Advanced Techniques
   to solve this

   This algorithm:
     1. searches each anime item for a match to the preferred title
        AND all synonyms and marks those matches with
          `synonym.length() - (synonym.find(needle) + needle.length());`
        which should never be less than zero and will be zero if, and only if
        the titles match exactly.
     2. returns the id of the match that is the lowest, which will most
        definitely match anything that exactly matches the title of the
        filename */
int Database::GetAnimeFromTitle(const std::string& title) {
	if (title.empty())
		return 0;

	std::unordered_map<int, size_t> map;

	auto process_title = [&map](const Anime& anime, const std::string& title, const std::string& needle) -> bool {
		size_t ret = title.find(needle);
		if (ret == std::string::npos)
			return false;

		map[anime.GetId()] = title.length() - (ret + needle.length());
		return true;
	};

	for (const auto& [id, anime] : items) {
		if (process_title(anime, anime.GetUserPreferredTitle(), title))
			continue;

		for (const auto& synonym : anime.GetTitleSynonyms())
			if (process_title(anime, synonym, title))
				continue;
	}

	return get_lowest_in_map(map);
}

static bool GetListDataAsJSON(const Anime& anime, nlohmann::json& json) {
	if (!anime.IsInUserList())
		return false;

	json = {
		{"status", Translate::ToString(anime.GetUserStatus())},
		{"progress", anime.GetUserProgress()},
		{"score", anime.GetUserScore()},
		//{"started", anime.GetUserDateStarted()},
		//{"completed", anime.GetUserDateCompleted()},
		{"private", anime.GetUserIsPrivate()},
		{"rewatched_times", anime.GetUserRewatchedTimes()},
		{"rewatching", anime.GetUserIsRewatching()},
		{"updated", anime.GetUserTimeUpdated()},
		{"notes", anime.GetUserNotes()}
	};
}

static bool GetAnimeAsJSON(const Anime& anime, nlohmann::json& json) {
	json = {
		{"id", anime.GetId()},
		{"title", {
			{"native", anime.GetNativeTitle()},
			{"romaji", anime.GetRomajiTitle()},
			{"english", anime.GetEnglishTitle()}
		}},
		{"synonyms", anime.GetTitleSynonyms()},
		{"episodes", anime.GetEpisodes()},
		{"airing_status", anime.GetAiringStatus()},
		{"air_date", anime.GetAirDate()},
		{"genres", anime.GetGenres()},
		{"producers", anime.GetProducers()},
		{"format", anime.GetFormat()},
		{"season", anime.GetSeason()},
		{"audience_score", anime.GetAudienceScore()},
		{"synopsis", anime.GetSynopsis()},
		{"duration", anime.GetDuration()},
		{"poster_url", anime.GetPosterUrl()},
		{"service_url", anime.GetServiceUrl()}
	};

	nlohmann::json user;
	if (GetListDataAsJSON(anime, user))
		json.push_back({"list_data", user});

	return true;
}

bool Database::GetDatabaseAsJSON(nlohmann::json& json) {
	for (const auto& [id, anime] : items) {
		nlohmann::json anime_json;
		GetAnimeAsJSON(anime, anime_json);
		json.push_back(anime_json);
	}
	return true;
}

Database db;

} // namespace Anime