#include <regex>

#include "animia.h"
#include "animia/fd.h"
#include "animia/strategies.h"
#include "animia/util.h"

namespace animia::internal {

class Strategist {
	public:
		Strategist(Result& result) : result_(result) {}

		bool ApplyStrategies();

	private:
		bool AddMedia(const MediaInfo media_information);

		bool ApplyOpenFilesStrategy();
		bool ApplyWindowTitleStrategy();

		Result& result_;
};

bool Strategist::ApplyStrategies() {
	bool success = false;

	switch (result_.type) {
		case ResultType::Process: success |= ApplyOpenFilesStrategy(); break;
		case ResultType::Window: success |= ApplyWindowTitleStrategy(); break;
	}

	return success;
}

bool ApplyStrategies(std::vector<Result>& results) {
	bool success = false;

	for (auto& result : results) {
		Strategist strategist(result);
		success |= strategist.ApplyStrategies();
	}

	return success;
}

////////////////////////////////////////////////////////////////////////////////

static bool ApplyWindowTitleFormat(const std::string& format, std::string& title) {
	if (format.empty())
		return false;

	const std::regex pattern(format);
	std::smatch match;
	std::regex_match(title, match, pattern);

	// Use the first non-empty match result, because the regular expression may
	// contain multiple sub-expressions.
	for (size_t i = 1; i < match.size(); ++i) {
		if (!match.str(i).empty()) {
			title = match.str(i);
			return true;
		}
	}

	// Results are empty, but the match was successful
	if (!match.empty()) {
		title.clear();
		return true;
	}

	return true;
}

static MediaInfoType InferMediaInformationType(const std::string& str) {
	const std::regex path_pattern(R"(^(?:[A-Za-z]:[/\\]|\\\\)[^<>:"/\\|?*]+)");
	return (std::regex_search(str, path_pattern)) ? MediaInfoType::File : MediaInfoType::Unknown;
}

bool Strategist::ApplyWindowTitleStrategy() {
	auto title = result_.window.text;
	ApplyWindowTitleFormat(result_.player.window_title_format, title);

	return AddMedia({InferMediaInformationType(title), title});
}

bool Strategist::ApplyOpenFilesStrategy() {
	bool success = false;

	const std::set<pid_t> pids{result_.process.pid};

	auto open_file_proc = [&](const OpenFile& file) -> bool {
		success |= AddMedia({MediaInfoType::File, file.path});
		return true;
	};

	fd.EnumerateOpenFiles(pids, open_file_proc);

	return success;
}

////////////////////////////////////////////////////////////////////////////////

bool Strategist::AddMedia(const MediaInfo media_information) {
	if (media_information.value.empty())
		return false;

	Media media;
	media.information.push_back(media_information);
	result_.media.push_back(std::move(media));

	return true;
}

} // namespace animia::internal
