#include "gui/widgets/text.h"
#include "core/session.h"
#include <QFrame>
#include <QLabel>
#include <QPixmap>
#include <QTextBlock>
#include <QVBoxLayout>
#include <QDebug>

namespace TextWidgets {

Header::Header(QString title, QWidget* parent) : QWidget(parent) {
	setLayout(new QVBoxLayout);
	setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum);

	static_text_title = new QLabel(title, this);
	static_text_title->setTextFormat(Qt::PlainText);
	QFont font = static_text_title->font();
	font.setWeight(QFont::Bold);
	static_text_title->setFont(font);
	static_text_title->setFixedHeight(16);

	static_text_line = new QFrame(this);
	static_text_line->setFrameShape(QFrame::HLine);
	static_text_line->setFrameShadow(QFrame::Sunken);
	static_text_line->setFixedHeight(2);

	layout()->addWidget(static_text_title);
	layout()->addWidget(static_text_line);
	layout()->setSpacing(0);
	layout()->setMargin(0);
}

void Header::SetTitle(QString title) {
	static_text_title->setText(title);
}

TextParagraph::TextParagraph(QString title, QString data, QWidget* parent) : QWidget(parent) {
	setLayout(new QVBoxLayout);

	header = new Header(title, this);

	QWidget* content = new QWidget(this);
	content->setLayout(new QHBoxLayout);

	paragraph = new Paragraph(data, this);
	paragraph->setTextInteractionFlags(Qt::NoTextInteraction);
	paragraph->setAttribute(Qt::WidgetAttribute::WA_TransparentForMouseEvents);
	paragraph->setWordWrapMode(QTextOption::NoWrap);

	content->layout()->addWidget(paragraph);
	content->layout()->setSpacing(0);
	content->layout()->setMargin(0);
	content->setContentsMargins(12, 0, 0, 0);

	layout()->addWidget(header);
	layout()->addWidget(paragraph);
	layout()->setSpacing(0);
	layout()->setMargin(0);
}

Header* TextParagraph::GetHeader() {
	return header;
}

Paragraph* TextParagraph::GetParagraph() {
	return paragraph;
}

LabelledTextParagraph::LabelledTextParagraph(QString title, QString label, QString data, QWidget* parent)
    : QWidget(parent) {
	setLayout(new QVBoxLayout);

	header = new Header(title, this);

	// this is not accessible from the object because there's really
	// no reason to make it accessible...
	QWidget* content = new QWidget(this);
	content->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred);

	labels = new Paragraph(label, this);
	labels->setTextInteractionFlags(Qt::NoTextInteraction);
	labels->setAttribute(Qt::WidgetAttribute::WA_TransparentForMouseEvents);
	labels->setWordWrapMode(QTextOption::NoWrap);
	labels->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred);
	labels->setFixedWidth(123);

	paragraph = new Paragraph(data, this);
	paragraph->setTextInteractionFlags(Qt::NoTextInteraction);
	paragraph->setAttribute(Qt::WidgetAttribute::WA_TransparentForMouseEvents);
	paragraph->setWordWrapMode(QTextOption::NoWrap);

	QHBoxLayout* content_layout = new QHBoxLayout;
	content_layout->addWidget(labels, 0, Qt::AlignTop);
	content_layout->addWidget(paragraph, 0, Qt::AlignTop);
	content_layout->setSpacing(0);
	content_layout->setMargin(0);
	content->setLayout(content_layout);

	content->setContentsMargins(12, 0, 0, 0);

	layout()->addWidget(header);
	layout()->addWidget(content);
	layout()->setSpacing(0);
	layout()->setMargin(0);
}

Header* LabelledTextParagraph::GetHeader() {
	return header;
}

Paragraph* LabelledTextParagraph::GetLabels() {
	return labels;
}

Paragraph* LabelledTextParagraph::GetParagraph() {
	return paragraph;
}

SelectableTextParagraph::SelectableTextParagraph(QString title, QString data, QWidget* parent) : QWidget(parent) {
	setLayout(new QVBoxLayout);

	header = new Header(title, this);

	QWidget* content = new QWidget(this);
	content->setLayout(new QHBoxLayout);

	paragraph = new Paragraph(data, content);

	content->layout()->addWidget(paragraph);
	content->layout()->setSpacing(0);
	content->layout()->setMargin(0);
	content->setContentsMargins(12, 0, 0, 0);

	layout()->addWidget(header);
	layout()->addWidget(content);
	layout()->setSpacing(0);
	layout()->setMargin(0);
}

Header* SelectableTextParagraph::GetHeader() {
	return header;
}

Paragraph* SelectableTextParagraph::GetParagraph() {
	return paragraph;
}

/* inherits QPlainTextEdit and gives a much more reasonable minimum size */
Paragraph::Paragraph(QString text, QWidget* parent) : QPlainTextEdit(text, parent) {
	setReadOnly(true);
	setTextInteractionFlags(Qt::TextBrowserInteraction);
	setFrameShape(QFrame::NoFrame);
	setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

	QPalette pal;
	pal.setColor(QPalette::Window, QColor(0, 0, 0, 0));
	setPalette(pal);

	setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
}

/* highly based upon... some stackoverflow answer for PyQt */
QSize Paragraph::minimumSizeHint() const {
	QTextDocument* doc = document();
	doc->adjustSize();
	long h = 0;
	for (QTextBlock line = doc->begin(); line != doc->end(); line = line.next()) {
		h += doc->documentLayout()->blockBoundingRect(line).height();
	}
	return QSize(QPlainTextEdit::sizeHint().width(), h);
}

QSize Paragraph::sizeHint() const {
	return minimumSizeHint();
}

/* this is still actually useful, so we'll keep it */
void SetPlainTextEditData(QPlainTextEdit* text_edit, QString data) {
	QTextDocument* document = new QTextDocument(text_edit);
	document->setDocumentLayout(new QPlainTextDocumentLayout(document));
	document->setPlainText(data);
	text_edit->setDocument(document);
}

} // namespace TextWidgets

#include "gui/widgets/moc_text.cpp"
