/**
 * vec - a tiny SIMD vector library in plain C99
 * 
 * Copyright (c) 2024 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

/* GCC built in vectors */

#include <stdint.h>
#include <string.h>

#define VEC_DEFINE_OPERATIONS(sign, csign, bits, size) \
	VEC_DECL_LOAD_ALIGNED(sign, bits, size) \
	{ \
		v##sign##int##bits##x##size vec; \
		memcpy(&vec, in, sizeof(*in) * size); \
		return vec; \
	} \
	\
	VEC_DECL_LOAD(sign, bits, size) \
	{ \
		return v##sign##int##bits##x##size##_load_aligned(in); \
	} \
	\
	VEC_DECL_STORE_ALIGNED(sign, bits, size) \
	{ \
		memcpy(out, &vec, sizeof(vec)); \
	} \
	\
	VEC_DECL_STORE(sign, bits, size) \
	{ \
		return v##sign##int##bits##x##size##_store_aligned(vec, out); \
	} \
	\
	VEC_DECL_ADD(sign, bits, size) \
	{ \
		return vec1 + vec2; \
	} \
	\
	VEC_DECL_SUB(sign, bits, size) \
	{ \
		return vec1 - vec2; \
	} \
	\
	VEC_DECL_MUL(sign, bits, size) \
	{ \
		return vec1 * vec2; \
	} \
	\
	VEC_DECL_AND(sign, bits, size) \
	{ \
		return vec1 & vec2; \
	} \
	\
	VEC_DECL_OR(sign, bits, size) \
	{ \
		return vec1 | vec2; \
	} \
	\
	VEC_DECL_XOR(sign, bits, size) \
	{ \
		return vec1 ^ vec2; \
	} \
	VEC_DECL_CMPLT(sign, bits, size) \
	{ \
		return vec1 < vec2; \
	} \
	VEC_DECL_CMPGT(sign, bits, size) \
	{ \
		return vec1 > vec2; \
	} \
	VEC_DECL_CMPEQ(sign, bits, size) \
	{ \
		return vec1 == vec2; \
	} \
	VEC_DECL_CMPLE(sign, bits, size) \
	{ \
		return vec1 <= vec2; \
	} \
	VEC_DECL_CMPGE(sign, bits, size) \
	{ \
		return vec1 >= vec2; \
	} \
	\
	VEC_GENERIC_DIVIDE(sign, csign, bits, size) \
	VEC_GENERIC_SPLAT(sign, csign, bits, size) \
	VEC_GENERIC_SHIFTS(sign, csign, bits, size) \
	VEC_GENERIC_AVG(sign, bits, size)

// -----------------------------------------------------------------------------------
// 128-bit vector types

#ifndef VEC_VUINT8X16
# define VEC_VUINT8X16
typedef uint8_t vuint8x16 __attribute__((__vector_size__(16)));
# define VUINT8x16_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) \
	(vuint8x16){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p }
# define VUINT8x16_ALIGNMENT 16
VEC_DEFINE_OPERATIONS(u, U, 8, 16)
#endif

#ifndef VEC_VUINT16X8
# define VEC_VUINT16X8
typedef uint16_t vuint16x8 __attribute__((__vector_size__(16)));
# define VUINT16x8_CONSTANT(a, b, c, d, e, f, g, h) \
	(vuint16x8){ a, b, c, d, e, f, g, h }
# define VUINT16x8_ALIGNMENT 16
VEC_DEFINE_OPERATIONS(u, U, 16, 8)
#endif

#ifndef VEC_VUINT32X4
# define VEC_VUINT32X4
typedef uint32_t vuint32x4 __attribute__((__vector_size__(16)));
# define VUINT32x4_CONSTANT(a, b, c, d) \
	(vuint32x4){ a, b, c, d }
# define VUINT32x4_ALIGNMENT 16
VEC_DEFINE_OPERATIONS(u, U, 32, 4)
#endif

#ifndef VEC_VUINT64X2
# define VEC_VUINT64X2
typedef uint64_t vuint64x2 __attribute__((__vector_size__(16)));
# define VUINT64x2_CONSTANT(a, b) \
	(vuint64x2){ a, b }
# define VUINT64x2_ALIGNMENT 16
VEC_DEFINE_OPERATIONS(u, U, 64, 2)
#endif

#ifndef VEC_VINT8X16
# define VEC_VINT8X16
typedef int8_t vint8x16 __attribute__((__vector_size__(16)));
# define VINT8x16_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) \
	(vint8x16){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p }
# define VINT8x16_ALIGNMENT 16
VEC_DEFINE_OPERATIONS(, , 8, 16)
#endif

#ifndef VEC_VINT16X8
# define VEC_VINT16X8
typedef int16_t vint16x8 __attribute__((__vector_size__(16)));
# define VINT16x8_CONSTANT(a, b, c, d, e, f, g, h) \
	(vint16x8){ a, b, c, d, e, f, g, h }
# define VINT16x8_ALIGNMENT 16
VEC_DEFINE_OPERATIONS(, , 16, 8)
#endif

#ifndef VEC_VINT32X4
# define VEC_VINT32X4
typedef int32_t vint32x4 __attribute__((__vector_size__(16)));
# define VINT32x4_CONSTANT(a, b, c, d) \
	(vint32x4){ a, b, c, d }
# define VINT32x4_ALIGNMENT 16
VEC_DEFINE_OPERATIONS(, , 32, 4)
#endif

#ifndef VEC_VINT64X2
# define VEC_VINT64X2
typedef int64_t vint64x2 __attribute__((__vector_size__(16)));
# define VINT64x2_CONSTANT(a, b) \
	(vint64x2){ a, b }
# define VINT64x2_ALIGNMENT 16
VEC_DEFINE_OPERATIONS(, , 64, 2)
#endif

#ifndef VEC_VUINT8X16
# define VEC_VUINT8X16
typedef uint8_t vuint8x16 __attribute__((__vector_size__(16)));
# define VUINT8x16_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) \
	(vuint8x16){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p }
# define VUINT8x16_ALIGNMENT 16
VEC_DEFINE_OPERATIONS(u, U, 8, 16)
#endif

#ifndef VEC_VUINT16X8
# define VEC_VUINT16X8
typedef uint16_t vuint16x8 __attribute__((__vector_size__(16)));
# define VUINT16x8_CONSTANT(a, b, c, d, e, f, g, h) \
	(vuint16x8){ a, b, c, d, e, f, g, h }
# define VUINT16x8_ALIGNMENT 16
VEC_DEFINE_OPERATIONS(u, U, 16, 8)
#endif

#ifndef VEC_VUINT32X4
# define VEC_VUINT32X4
typedef uint32_t vuint32x4 __attribute__((__vector_size__(16)));
# define VUINT32x4_CONSTANT(a, b, c, d) \
	(vuint32x4){ a, b, c, d }
# define VUINT32x4_ALIGNMENT 16
VEC_DEFINE_OPERATIONS(u, U, 32, 4)
#endif

#ifndef VEC_VUINT64X2
# define VEC_VUINT64X2
typedef uint64_t vuint64x2 __attribute__((__vector_size__(16)));
# define VUINT64x2_CONSTANT(a, b) \
	(vuint64x2){ a, b }
# define VUINT64x2_ALIGNMENT 16
VEC_DEFINE_OPERATIONS(u, U, 64, 2)
#endif

// --------------------------------------------------------------------------
// 256-bit vector types

#ifndef VEC_VUINT8X32
# define VEC_VUINT8X32
typedef uint8_t vuint8x32 __attribute__((__vector_size__(32)));
# define VUINT8x32_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, u, v, w, x, y, z, aa, ab, ac, ad, ae, af) \
	((vuint8x16){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, u, v, w, x, y, z, aa, ab, ac, ad, ae, af })
# define VUINT8x32_ALIGNMENT 32
VEC_DEFINE_OPERATIONS(u, U, 8, 32)
#endif

#ifndef VEC_VUINT16X16
# define VEC_VUINT16X16
typedef uint16_t vuint16x16 __attribute__((__vector_size__(32)));
# define VUINT16x16_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) \
	(vuint16x16){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p }
# define VUINT16x16_ALIGNMENT 32
VEC_DEFINE_OPERATIONS(u, U, 16, 16)
#endif

#ifndef VEC_VUINT32X8
# define VEC_VUINT32X8
typedef uint32_t vuint32x8 __attribute__((__vector_size__(32)));
# define VUINT32x8_CONSTANT(a, b, c, d, e, f, g, h) \
	(vuint32x8){ a, b, c, d, e, f, g, h }
# define VUINT32x8_ALIGNMENT 32
VEC_DEFINE_OPERATIONS(u, U, 32, 8)
#endif

#ifndef VEC_VUINT64X4
# define VEC_VUINT64X4
typedef uint64_t vuint64x4 __attribute__((__vector_size__(32)));
# define VUINT64x4_CONSTANT(a, b, c, d) \
	(vuint64x4){ a, b, c, d }
# define VUINT64x4_ALIGNMENT 32
VEC_DEFINE_OPERATIONS(u, U, 64, 4)
#endif

#ifndef VEC_VINT8X32
# define VEC_VINT8X32
typedef int8_t vint8x32 __attribute__((__vector_size__(32)));
# define VINT8x32_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, u, v, w, x, y, z, aa, ab, ac, ad, ae, af) \
	((vint8x16){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, u, v, w, x, y, z, aa, ab, ac, ad, ae, af })
# define VINT8x32_ALIGNMENT 32
VEC_DEFINE_OPERATIONS(, , 8, 32)
#endif

#ifndef VEC_VINT16X16
# define VEC_VINT16X16
typedef int16_t vint16x16 __attribute__((__vector_size__(32)));
# define VINT16x16_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) \
	(vint16x16){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p }
# define VINT16x16_ALIGNMENT 32
VEC_DEFINE_OPERATIONS(, , 16, 16)
#endif

#ifndef VEC_VINT32X8
# define VEC_VINT32X8
typedef int32_t vint32x8 __attribute__((__vector_size__(32)));
# define VINT32x8_CONSTANT(a, b, c, d, e, f, g, h) \
	(vint32x8){ a, b, c, d, e, f, g, h }
# define VINT32x8_ALIGNMENT 32
VEC_DEFINE_OPERATIONS(, , 32, 8)
#endif

#ifndef VEC_VINT64X4
# define VEC_VINT64X4
typedef int64_t vint64x4 __attribute__((__vector_size__(32)));
# define VINT64x4_CONSTANT(a, b, c, d) \
	(vint64x4){ a, b, c, d }
# define VINT64x4_ALIGNMENT 32
VEC_DEFINE_OPERATIONS(, , 64, 4)
#endif

// --------------------------------------------------------------------------
// 512-bit vector types

#ifndef VEC_VUINT8X64
# define VEC_VUINT8X64
typedef uint8_t vuint8x64 __attribute__((__vector_size__(64)));
# define VUINT8x64_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, u, v, w, x, y, z, aa, ab, ac, ad, ae, af, ag, ah, ai, aj, ak, al, am, an, ao, ap, aq, ar, as, at, au, av, aw, ax, ay, az, ba, bb, bc, bd, be, bf, bg, bh, bi, bj, bk, bl) \
	((vuint8x64){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, u, v, w, x, y, z, aa, ab, ac, ad, ae, af, ag, ah, ai, aj, ak, al, am, an, ao, ap, aq, ar, as, at, au, av, aw, ax, ay, az, ba, bb, bc, bd, be, bf, bg, bh, bi, bj, bk, bl })
# define VUINT8x64_ALIGNMENT 64
VEC_DEFINE_OPERATIONS(u, U, 8, 64)
#endif

#ifndef VEC_VUINT16X32
# define VEC_VUINT16X32
typedef uint16_t vuint16x32 __attribute__((__vector_size__(64)));
# define VUINT16x32_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, u, v, w, x, y, z, aa, ab, ac, ad, ae, af) \
	((vuint16x32){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, u, v, w, x, y, z, aa, ab, ac, ad, ae, af })
# define VUINT16x32_ALIGNMENT 64
VEC_DEFINE_OPERATIONS(u, U, 16, 32)
#endif

#ifndef VEC_VUINT32X16
# define VEC_VUINT32X16
typedef uint32_t vuint32x16 __attribute__((__vector_size__(64)));
# define VUINT32x16_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) \
	(vuint32x16){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p }
# define VUINT32x16_ALIGNMENT 64
VEC_DEFINE_OPERATIONS(u, U, 32, 16)
#endif

#ifndef VEC_VUINT64X8
# define VEC_VUINT64X8
typedef uint64_t vuint64x8 __attribute__((__vector_size__(64)));
# define VUINT64x8_CONSTANT(a, b, c, d, e, f, g, h) \
	(vuint64x8){ a, b, c, d, e, f, g, h }
# define VUINT64x8_ALIGNMENT 64
VEC_DEFINE_OPERATIONS(u, U, 64, 8)
#endif

#ifndef VEC_VINT8X64
# define VEC_VINT8X64
typedef int8_t vint8x64 __attribute__((__vector_size__(64)));
# define VINT8x64_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, u, v, w, x, y, z, aa, ab, ac, ad, ae, af, ag, ah, ai, aj, ak, al, am, an, ao, ap, aq, ar, as, at, au, av, aw, ax, ay, az, ba, bb, bc, bd, be, bf, bg, bh, bi, bj, bk, bl) \
	((vint8x64){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, u, v, w, x, y, z, aa, ab, ac, ad, ae, af, ag, ah, ai, aj, ak, al, am, an, ao, ap, aq, ar, as, at, au, av, aw, ax, ay, az, ba, bb, bc, bd, be, bf, bg, bh, bi, bj, bk, bl })
# define VINT8x64_ALIGNMENT 64
VEC_DEFINE_OPERATIONS(, , 8, 64)
#endif

#ifndef VEC_VINT16X32
# define VEC_VINT16X32
typedef int16_t vint16x32 __attribute__((__vector_size__(64)));
# define VINT16x32_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, u, v, w, x, y, z, aa, ab, ac, ad, ae, af) \
	((vint16x32){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, u, v, w, x, y, z, aa, ab, ac, ad, ae, af })
# define VINT16x32_ALIGNMENT 64
VEC_DEFINE_OPERATIONS(, , 16, 32)
#endif

#ifndef VEC_VINT32X16
# define VEC_VINT32X16
typedef int32_t vint32x16 __attribute__((__vector_size__(64)));
# define VINT32x16_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) \
	(vint32x16){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p }
# define VINT32x16_ALIGNMENT 64
VEC_DEFINE_OPERATIONS(, , 32, 16)
#endif

#ifndef VEC_VINT64X8
# define VEC_VINT64X8
typedef int64_t vint64x8 __attribute__((__vector_size__(64)));
# define VINT64x8_CONSTANT(a, b, c, d, e, f, g, h) \
	(vint64x8){ a, b, c, d, e, f, g, h }
# define VINT64x8_ALIGNMENT 64
VEC_DEFINE_OPERATIONS(, , 64, 8)
#endif

// ----------------------------------------------------------

#undef VEC_DEFINE_OPERATIONS
