/**
 * vec - a tiny SIMD vector library in plain C99
 * 
 * Copyright (c) 2024 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

/* Altivec vector support. */

#include <stdint.h>
#include <string.h>

#include <altivec.h>

#define VEC_ALTIVEC_ALIGNMENT 16

/* Since altivec conveniently made their API super user friendly, we can just use
 * one giant macro to define literally everything */
#define VEC_DEFINE_OPERATIONS(sign, bits, size) \
	static inline VEC_ALWAYS_INLINE v##sign##int##bits##x##size v##sign##int##bits##x##size##_splat(sign##int##bits##_t i) \
	{ \
		return vec_splats(i); \
	} \
	\
	static inline VEC_ALWAYS_INLINE v##sign##int##bits##x##size v##sign##int##bits##x##size##_load(const sign##int##bits##_t in[size]) \
	{ \
		return vec_perm(vec_ld(0, in), vec_ld(VEC_ALTIVEC_ALIGNMENT, in), vec_lvsl(0, in)); \
	} \
	\
	static inline VEC_ALWAYS_INLINE void v##sign##int##bits##x##size##_store(v##sign##int##bits##x##size vec, sign##int##bits##_t out[size]) \
	{ \
		VEC_ALIGNED_ARRAY(sign##int##bits##_t, aligned_out, size, VEC_ALTIVEC_ALIGNMENT); \
		vec_st(vec, 0, aligned_out); \
		memcpy(out, aligned_out, size * sizeof(*aligned_out)); \
	} \
	\
	static inline VEC_ALWAYS_INLINE v##sign##int##bits##x##size v##sign##int##bits##x##size##_add(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return vec_add(vec1, vec2); \
	} \
	\
	static inline VEC_ALWAYS_INLINE v##sign##int##bits##x##size v##sign##int##bits##x##size##_sub(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return vec_sub(vec1, vec2); \
	} \
	\
	static inline VEC_ALWAYS_INLINE v##sign##int##bits##x##size v##sign##int##bits##x##size##_mul(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return vec_mul(vec1, vec2); \
	}

#ifndef VEC_VUINT8X16
# define VEC_VUINT8X16
typedef vector unsigned char vuint8x16;
# define VUINT8x16_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) \
	(vuint8x16){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p }
VEC_DEFINE_OPERATIONS(u, 8, 16)
# define VUINT8x16_ALIGNED(x) ((x) % VEC_ALTIVEC_ALIGNMENT == 0)
#endif /* VEC_VUINT8X16 */

#ifndef VEC_VINT8X16
# define VEC_VINT8X16
typedef vector signed char vint8x16;
# define VINT8x16_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) \
	(vint8x16){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p }
VEC_DEFINE_OPERATIONS(, 8, 16)
# define VINT8x16_ALIGNED(x) ((x) % VEC_ALTIVEC_ALIGNMENT == 0)
#endif /* VEC_VINT8X16 */

#ifndef VEC_VUINT16X8
# define VEC_VUINT16X8
typedef vector unsigned short vuint16x8;
# define VUINT16x8_CONSTANT(a, b, c, d, e, f, g, h) \
	(vuint16x8){ a, b, c, d, e, f, g, h }
VEC_DEFINE_OPERATIONS(u, 16, 8)
# define VUINT16x8_ALIGNED(x) ((x) % VEC_ALTIVEC_ALIGNMENT == 0)
#endif /* VEC_VUINT16X8 */

#ifndef VEC_VINT16X8
# define VEC_VINT16X8
typedef vector signed short vint16x8;
# define VINT16x8_CONSTANT(a, b, c, d, e, f, g, h) \
	(vint16x8){ a, b, c, d, e, f, g, h }
VEC_DEFINE_OPERATIONS(, 16, 8)
# define VINT16x8_ALIGNED(x) ((x) % VEC_ALTIVEC_ALIGNMENT == 0)
#endif /* VEC_VINT16X8 */

#ifndef VEC_VUINT32X4
# define VEC_VUINT32X4
typedef vector unsigned int vuint32x4;
# define VUINT32x4_CONSTANT(a, b, c, d) \
	(vuint32x4){ a, b, c, d }
VEC_DEFINE_OPERATIONS(u, 32, 4)
# define VUINT32x4_ALIGNED(x) ((x) % VEC_ALTIVEC_ALIGNMENT == 0)
#endif /* VEC_VUINT32X4 */

#ifndef VEC_VINT32X4
# define VEC_VINT32X4
typedef vector signed int vint32x4;
# define VINT32x4_CONSTANT(a, b, c, d) \
	(vint32x4){ a, b, c, d }
VEC_DEFINE_OPERATIONS(, 32, 4)
# define VINT32x4_ALIGNED(x) ((x) % VEC_ALTIVEC_ALIGNMENT == 0)
#endif /* VEC_VINT32X4 */

#if defined(__POWER8__) && defined(__VSX__)

# ifndef VEC_VUINT64X2
#  define VEC_VUINT64X2
typedef vector unsigned long long vuint64x2;
#  define VUINT64x2_CONSTANT(a, b) \
	(vuint64x2){ a, b }
VEC_DEFINE_OPERATIONS(u, 64, 2)
#  define VUINT64x2_ALIGNED(x) ((x) % VEC_ALTIVEC_ALIGNMENT == 0)
# endif /* VEC_VUINT64X2 */

# ifndef VEC_VINT64X2
#  define VEC_VINT64X2
typedef vector signed long long vint64x2;
#  define VINT64x2_CONSTANT(a, b) \
	(vint64x2){ a, b }
VEC_DEFINE_OPERATIONS(, 64, 2)
#  define VINT64x2_ALIGNED(x) ((x) % VEC_ALTIVEC_ALIGNMENT == 0)
# endif /* VEC_VINT64X2 */

#endif /* defined(__POWER8__) && defined(__VSX__) */

#undef VEC_DEFINE_OPERATIONS
