/**
 * vec - a tiny SIMD vector library in plain C99
 * 
 * Copyright (c) 2024 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

/* GCC built in vectors */

#include <stdint.h>
#include <string.h>

#define VEC_DEFINE_OPERATIONS(sign, bits, size) \
	VEC_DECL_LOAD_ALIGNED(sign, bits, size) \
	{ \
		v##sign##int##bits##x##size vec; \
		memcpy(&vec, in, sizeof(vec)); \
		return vec; \
	} \
	\
	VEC_DECL_LOAD(sign, bits, size) \
	{ \
		return v##sign##int##bits##x##size##_load_aligned(in); \
	} \
	\
	VEC_DECL_STORE_ALIGNED(sign, bits, size) \
	{ \
		memcpy(out, &vec, sizeof(vec)); \
	} \
	\
	VEC_DECL_STORE(sign, bits, size) \
	{ \
		return v##sign##int##bits##x##size##_store_aligned(vec, out); \
	} \
	\
	VEC_DECL_ADD(sign, bits, size) \
	{ \
		return vec1 + vec2; \
	} \
	\
	VEC_DECL_SUB(sign, bits, size) \
	{ \
		return vec1 - vec2; \
	} \
	\
	VEC_DECL_MUL(sign, bits, size) \
	{ \
		return vec1 * vec2; \
	} \
	\
	VEC_DECL_AND(sign, bits, size) \
	{ \
		return vec1 & vec2; \
	} \
	\
	VEC_DECL_OR(sign, bits, size) \
	{ \
		return vec1 | vec2; \
	} \
	\
	VEC_DECL_XOR(sign, bits, size) \
	{ \
		return vec1 ^ vec2; \
	} \
	VEC_DECL_CMPLT(sign, bits, size) \
	{ \
		return vec1 < vec2; \
	} \
	VEC_DECL_CMPGT(sign, bits, size) \
	{ \
		return vec1 > vec2; \
	} \
	VEC_DECL_CMPEQ(sign, bits, size) \
	{ \
		return vec1 == vec2; \
	} \
	VEC_DECL_CMPLE(sign, bits, size) \
	{ \
		return vec1 <= vec2; \
	} \
	VEC_DECL_CMPGE(sign, bits, size) \
	{ \
		return vec1 >= vec2; \
	} \
	\
	VEC_GENERIC_DIVIDE(sign, bits, size) \
	VEC_GENERIC_SPLAT(sign, bits, size) \
	VEC_GENERIC_SHIFTS(sign, bits, size) \
	VEC_GENERIC_AVG(sign, bits, size)

#ifndef VEC_VUINT8X16
# define VEC_VUINT8X16
typedef uint8_t vuint8x16 __attribute__((__vector_size__(16)));
# define VUINT8x16_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) \
	(vuint8x16){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p }
VEC_DEFINE_OPERATIONS(u, 8, 16)
# define VINT8x16_ALIGNED 1
#endif

#ifndef VEC_VUINT16X8
# define VEC_VUINT16X8
typedef uint16_t vuint16x8 __attribute__((__vector_size__(16)));
# define VUINT16x8_CONSTANT(a, b, c, d, e, f, g, h) \
	(vuint16x8){ a, b, c, d, e, f, g, h }
VEC_DEFINE_OPERATIONS(u, 16, 8)
# define VINT16x8_ALIGNED 1
#endif

#ifndef VEC_VUINT32X4
# define VEC_VUINT32X4
typedef uint32_t vuint32x4 __attribute__((__vector_size__(16)));
# define VUINT32x4_CONSTANT(a, b, c, d) \
	(vuint32x4){ a, b, c, d }
VEC_DEFINE_OPERATIONS(u, 32, 4)
# define VINT32x4_ALIGNED 1
#endif

#ifndef VEC_VUINT64X2
# define VEC_VUINT64X2
typedef uint64_t vuint64x2 __attribute__((__vector_size__(16)));
# define VUINT64x2_CONSTANT(a, b) \
	(vuint64x2){ a, b }
VEC_DEFINE_OPERATIONS(u, 64, 2)
# define VINT64x2_ALIGNED 1
#endif

#ifndef VEC_VINT8X16
# define VEC_VINT8X16
typedef int8_t vint8x16 __attribute__((__vector_size__(16)));
# define VINT8x16_CONSTANT(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) \
	(vint8x16){ a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p }
VEC_DEFINE_OPERATIONS(, 8, 16)
# define VINT8x16_ALIGNED 1
#endif

#ifndef VEC_VINT16X8
# define VEC_VINT16X8
typedef int16_t vint16x8 __attribute__((__vector_size__(16)));
# define VINT16x8_CONSTANT(a, b, c, d, e, f, g, h) \
	(vint16x8){ a, b, c, d, e, f, g, h }
VEC_DEFINE_OPERATIONS(, 16, 8)
# define VINT16x8_ALIGNED 1
#endif

#ifndef VEC_VINT32X4
# define VEC_VINT32X4
typedef int32_t vint32x4 __attribute__((__vector_size__(16)));
# define VINT32x4_CONSTANT(a, b, c, d) \
	(vint32x4){ a, b, c, d }
VEC_DEFINE_OPERATIONS(, 32, 4)
# define VINT32x4_ALIGNED 1
#endif

#ifndef VEC_VINT64X2
# define VEC_VINT64X2
typedef int64_t vint64x2 __attribute__((__vector_size__(16)));
# define VINT64x2_CONSTANT(a, b) \
	(vint64x2){ a, b }
VEC_DEFINE_OPERATIONS(, 64, 2)
# define VINT64x2_ALIGNED 1
#endif

#undef VEC_DEFINE_OPERATIONS
