/**
 * vec - a tiny SIMD vector library in C99
 * 
 * Copyright (c) 2024 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

#ifndef VEC_IMPL_X86_AVX2_H_
#define VEC_IMPL_X86_AVX2_H_

#define VEC_AVX2_OPERATION_8x32_16x16(op, sign) \
	do { \
		/* unpack and multiply */ \
		__m256i dst_even = _mm256_##op##_epi16(vec1.avx2, vec2.avx2); \
		__m256i dst_odd = _mm256_##op##_epi16(_mm256_srli_epi16(vec1.avx2, 8), _mm256_srli_epi16(vec2.avx2, 8)); \
	\
		/* repack */ \
		v##sign##int8x32 vec; \
		vec.avx2 = _mm256_or_si256( \
			_mm256_slli_epi16(dst_odd, 8), \
			_mm256_srli_epi16(_mm256_slli_epi16(dst_even, 8), 8) \
		); \
		return vec; \
	} while (0)

#define VEC_AVX2_OPERATION_8x32_32x8(op, sign) \
	do { \
		/* unpack */ \
		__m256i dst_1 = _mm256_##op##_epi32(vec1.avx2, vec2.avx2); \
		__m256i dst_2 = _mm256_##op##_epi32(_mm256_srli_epi32(vec1.avx2, 8), _mm256_srli_epi32(vec2.avx2, 8)); \
		__m256i dst_3 = _mm256_##op##_epi32(_mm256_srli_epi32(vec1.avx2, 16), _mm256_srli_epi32(vec2.avx2, 16)); \
		__m256i dst_4 = _mm256_##op##_epi32(_mm256_srli_epi32(vec1.avx2, 24), _mm256_srli_epi32(vec2.avx2, 24)); \
	\
		/* repack */ \
		v##sign##int8x32 vec; \
		vec.avx2 = _mm256_or_si256( \
			_mm256_or_si256( \
				_mm256_slli_epi32(dst_4, 8), \
				_mm256_srli_epi32(_mm256_slli_epi32(dst_3, 8), 8) \
			), \
			_mm256_or_si256( \
				_mm256_slli_epi32(_mm256_slli_epi32(dst_2, 8), 16), \
				_mm256_srli_epi32(_mm256_slli_epi32(dst_1, 8), 24) \
			) \
		); \
		return vec; \
	} while (0)

#define VEC_AVX2_OPERATION_16x16(op, sign) \
	do { \
		/* unpack and multiply */ \
		__m256i dst_even = _mm256_##op##_epi32(vec1.avx2, vec2.avx2); \
		__m256i dst_odd = _mm256_##op##_epi32(_mm256_srli_epi32(vec1.avx2, 16), _mm256_srli_epi32(vec2.avx2, 16)); \
	\
		/* repack */ \
		v##sign##int16x16 vec; \
		vec.avx2 = _mm256_or_si256( \
			_mm256_slli_epi32(dst_odd, 16), \
			_mm256_srli_epi32(_mm256_slli_epi16(dst_even, 16), 16) \
		); \
		return vec; \
	} while (0)

// shifting

#define VEC_AVX2_LSHIFT_8x32(sign) \
	VEC_AVX2_OPERATION_8x32_32x8(sllv, sign)

#define VEC_AVX2_LSHIFT_16x16(sign) \
	VEC_AVX2_OPERATION_16x16(sllv, sign)

#define VEC_AVX2_LSHIFT_32x8(sign) \
	do { \
		v##sign##int32x8 vec; \
		vec.avx2 = _mm256_sllv_epi32(vec1.avx2, vec2.avx2); \
		return vec; \
	} while (0)

#define VEC_AVX2_LSHIFT_64x4(sign) \
	do { \
		v##sign##int64x4 vec; \
		vec.avx2 = _mm256_sllv_epi64(vec1.avx2, vec2.avx2); \
		return vec; \
	} while (0)

#define VEC_AVX2_RSHIFT_8x32(sign, aORl) \
	VEC_AVX2_OPERATION_8x32_32x8(sr##aORl##v, sign)

#define VEC_AVX2_RSHIFT_16x16(sign, aORl) \
	VEC_AVX2_OPERATION_16x16(sr##aORl##v, sign)

#define VEC_AVX2_RSHIFT_32x8(sign, aORl) \
	do { \
		v##sign##int32x8 vec; \
		vec.avx2 = _mm256_sr##aORl##v_epi32(vec1.avx2, vec2.avx2); \
		return vec; \
	} while (0)

#define VEC_AVX2_aRSHIFT_64x4(sign) \
	do { \
		return v##sign##int64x4_fallback_rshift(vec1, vec2); \
	} while (0)

#define VEC_AVX2_lRSHIFT_64x4(sign) \
	do { \
		v##sign##int64x4 vec; \
		vec.avx2 = _mm256_srlv_epi64(vec1.avx2, vec2.avx2); \
		return vec; \
	} while (0)

#define VEC_AVX2_RSHIFT_64x4(sign, aORl) \
	VEC_AVX2_##aORl##RSHIFT_64x4(sign)

// multiplication

#define VEC_AVX2_MUL_8x32(sign) \
	VEC_AVX2_OPERATION_8x32_16x16(mullo, sign)

#define VEC_AVX2_MUL_16x16(sign) \
	do { \
		v##sign##int16x16 vec; \
		vec.avx2 = _mm256_mullo_epi16(vec1.avx2, vec2.avx2); \
		return vec; \
	} while (0)

#define VEC_AVX2_MUL_32x8(sign) \
	do { \
		v##sign##int32x8 vec; \
		vec.avx2 = _mm256_mullo_epi32(vec1.avx2, vec2.avx2); \
		return vec; \
	} while (0)

#define VEC_AVX2_MUL_64x4(sign) \
	do { \
		__m256i ac = _mm256_mul_epu32(vec1.avx2, vec2.avx2); \
		__m256i b  = _mm256_srli_epi64(vec1.avx2, 32); \
		__m256i bc = _mm256_mul_epu32(b, vec2.avx2); \
		__m256i d  = _mm256_srli_epi64(vec2.avx2, 32); \
		__m256i ad = _mm256_mul_epu32(vec1.avx2, d); \
		__m256i hi = _mm256_add_epi64(bc, ad); \
		hi = _mm256_slli_epi64(hi, 32); \
	\
		v##sign##int64x4 vec; \
		vec.avx2 = _mm256_add_epi64(hi, ac); \
		return vec; \
	} while (0)

// operations

#define VEC_AVX2_DEFINE_OPERATIONS_SIGN(sign, bits, size) \
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx2_load_aligned(const vec_##sign##int##bits in[size]) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.avx2 = _mm256_load_si256((const __m256i *)in); \
		return vec; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx2_load(const vec_##sign##int##bits in[size]) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.avx2 = _mm256_loadu_si256((const __m256i *)in); \
		return vec; \
	} \
	\
	static void v##sign##int##bits##x##size##_avx2_store_aligned(v##sign##int##bits##x##size vec, vec_##sign##int##bits out[size]) \
	{ \
		_mm256_store_si256((__m256i *)out, vec.avx2); \
	} \
	\
	static void v##sign##int##bits##x##size##_avx2_store(v##sign##int##bits##x##size vec, vec_##sign##int##bits out[size]) \
	{ \
		_mm256_storeu_si256((__m256i *)out, vec.avx2); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx2_add(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.avx2 = _mm256_add_epi##bits(vec1.avx2, vec2.avx2); \
		return vec; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx2_sub(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.avx2 = _mm256_sub_epi##bits(vec1.avx2, vec2.avx2); \
		return vec; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx2_mul(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		VEC_AVX2_MUL_##bits##x##size(sign); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx2_and(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.avx2 = _mm256_and_si256(vec1.avx2, vec2.avx2); \
		return vec; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx2_or(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.avx2 = _mm256_or_si256(vec1.avx2, vec2.avx2); \
		return vec; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx2_xor(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.avx2 = _mm256_xor_si256(vec1.avx2, vec2.avx2); \
		return vec; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx2_lshift(v##sign##int##bits##x##size vec1, vuint##bits##x##size vec2) \
	{ \
		VEC_AVX2_LSHIFT_##bits##x##size(sign); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx2_rshift(v##sign##int##bits##x##size vec1, vuint##bits##x##size vec2) \
	{ \
		VEC_AVX2_RSHIFT_##bits##x##size(sign, a); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx2_lrshift(v##sign##int##bits##x##size vec1, vuint##bits##x##size vec2) \
	{ \
		VEC_AVX2_RSHIFT_##bits##x##size(sign, l); \
	} \
	\
	static v##sign##int##bits##x##size##_impl v##sign##int##bits##x##size##_impl_avx2 = { \
		/* .splat = */ NULL, \
		v##sign##int##bits##x##size##_avx2_load_aligned, \
		v##sign##int##bits##x##size##_avx2_load, \
		v##sign##int##bits##x##size##_avx2_store_aligned, \
		v##sign##int##bits##x##size##_avx2_store, \
		v##sign##int##bits##x##size##_avx2_add, \
		v##sign##int##bits##x##size##_avx2_sub, \
		v##sign##int##bits##x##size##_avx2_mul, \
		/* .div = */ NULL, \
		/* .avg = */ NULL, \
		v##sign##int##bits##x##size##_avx2_and, \
		v##sign##int##bits##x##size##_avx2_or, \
		v##sign##int##bits##x##size##_avx2_xor, \
		/* .not = */ NULL, \
		v##sign##int##bits##x##size##_avx2_lshift, \
		v##sign##int##bits##x##size##_avx2_rshift, \
		v##sign##int##bits##x##size##_avx2_lrshift, \
	};

#define VEC_AVX2_DEFINE_OPERATIONS(bits, size) \
	VEC_AVX2_DEFINE_OPERATIONS_SIGN( , bits, size) \
	VEC_AVX2_DEFINE_OPERATIONS_SIGN(u, bits, size)

VEC_AVX2_DEFINE_OPERATIONS(8, 32)
VEC_AVX2_DEFINE_OPERATIONS(16, 16)
VEC_AVX2_DEFINE_OPERATIONS(32, 8)
VEC_AVX2_DEFINE_OPERATIONS(64, 4)

#undef VEC_AVX2_DEFINE_OPERATIONS
#undef VEC_AVX2_DEFINE_OPERATIONS_SIGN
#undef VEC_AVX2_MUL_8x32
#undef VEC_AVX2_MUL_16x16
#undef VEC_AVX2_MUL_32x8
#undef VEC_AVX2_MUL_64x4
#undef VEC_AVX2_OPERATION_8x32_16x16
#undef VEC_AVX2_OPERATION_8x32_32x8
#undef VEC_AVX2_OPERATION_16x16
#undef VEC_AVX2_LSHIFT_8x32
#undef VEC_AVX2_LSHIFT_16x16
#undef VEC_AVX2_LSHIFT_32x8
#undef VEC_AVX2_LSHIFT_64x4
#undef VEC_AVX2_RSHIFT_8x32
#undef VEC_AVX2_RSHIFT_16x16
#undef VEC_AVX2_RSHIFT_32x8
#undef VEC_AVX2_aRSHIFT_64x4
#undef VEC_AVX2_lRSHIFT_64x4
#undef VEC_AVX2_RSHIFT_64x4

#endif /* VEC_IMPL_X86_AVX2_H_ */
