/**
 * vec - a tiny SIMD vector library in C99
 * 
 * Copyright (c) 2024-2025 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

#ifndef VEC_IMPL_X86_AVX512F_H_
#define VEC_IMPL_X86_AVX512F_H_

#define VEC_AVX512F_OPERATION_EX_EX(name, op, sign, bits, size, secondsign, intlsign) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_##name(v##sign##int##bits##x##size vec1, v##secondsign##int##bits##x##size vec2) \
	{ \
		vec1.avx512f = _mm512_##op##_ep##intlsign##bits(vec1.avx512f, vec2.avx512f); \
	\
		return vec1; \
	}

#define VEC_AVX512F_MINMAX_TEMPLATE(SIGN, BITS, SIZE, INTLSIGN, OP) \
	VEC_AVX512F_OPERATION_EX_EX(OP, OP, SIGN, BITS, SIZE, SIGN, INTLSIGN)

#ifndef VINT32x16_MIN_DEFINED
VEC_AVX512F_MINMAX_TEMPLATE( , 32, 16, i, min)
# define VINT32x16_MIN_DEFINED
#endif

#ifndef VINT32x16_MAX_DEFINED
VEC_AVX512F_MINMAX_TEMPLATE( , 32, 16, i, max)
# define VINT32x16_MAX_DEFINED
#endif

#ifndef VUINT32x16_MIN_DEFINED
VEC_AVX512F_MINMAX_TEMPLATE(u, 32, 16, u, min)
# define VUINT32x16_MIN_DEFINED
#endif

#ifndef VUINT32x16_MAX_DEFINED
VEC_AVX512F_MINMAX_TEMPLATE(u, 32, 16, u, max)
# define VUINT32x16_MAX_DEFINED
#endif

#ifndef VINT64x8_MIN_DEFINED
VEC_AVX512F_MINMAX_TEMPLATE( , 64, 8, i, min)
# define VINT64x8_MIN_DEFINED
#endif

#ifndef VINT64x8_MAX_DEFINED
VEC_AVX512F_MINMAX_TEMPLATE( , 64, 8, i, max)
# define VINT64x8_MAX_DEFINED
#endif

#ifndef VUINT64x8_MIN_DEFINED
VEC_AVX512F_MINMAX_TEMPLATE(u, 64, 8, u, min)
# define VUINT64x8_MIN_DEFINED
#endif

#ifndef VUINT64x8_MAX_DEFINED
VEC_AVX512F_MINMAX_TEMPLATE(u, 64, 8, u, max)
# define VUINT64x8_MAX_DEFINED
#endif

/* ------------------------------------------------------------------------ */

#define VEC_AVX512F_OPERATION_EX(name, op, sign, bits, size, secondsign) \
	VEC_AVX512F_OPERATION_EX_EX(name, op, sign, bits, size, secondsign, i)

#define VEC_AVX512F_OPERATION(name, op, sign, bits, size) \
	VEC_AVX512F_OPERATION_EX(name, op, sign, bits, size, sign)

#define VEC_AVX512F_OPERATION_SHIFT(name, op, sign, bits, size) \
	VEC_AVX512F_OPERATION_EX(name, op, sign, bits, size, u)

#define VEC_AVX512F_XRSHIFT(name, bits, size, sign, aORl) \
	VEC_AVX512F_OPERATION_SHIFT(name, sr##aORl##v, sign, bits, size)

/* bitshift */
#ifndef VINT32x16_LRSHIFT_DEFINED
VEC_AVX512F_XRSHIFT(lrshift, 32, 16, /* nothing */, l)
# define VINT32x16_LRSHIFT_DEFINED
#endif

#ifndef VINT64x8_LRSHIFT_DEFINED
VEC_AVX512F_XRSHIFT(lrshift, 64, 8, /* nothing */, l)
# define VINT64x8_LRSHIFT_DEFINED
#endif

#ifndef VUINT32x16_LRSHIFT_DEFINED
VEC_AVX512F_XRSHIFT(lrshift, 32, 16, u, l)
# define VUINT32x16_LRSHIFT_DEFINED
#endif

#ifndef VUINT64x8_LRSHIFT_DEFINED
VEC_AVX512F_XRSHIFT(lrshift, 64, 8, u, l)
# define VUINT64x8_LRSHIFT_DEFINED
#endif

#ifndef VINT32x16_RSHIFT_DEFINED
VEC_AVX512F_XRSHIFT(rshift, 32, 16, /* nothing */, a)
# define VINT32x16_RSHIFT_DEFINED
#endif

#ifndef VINT64x8_RSHIFT_DEFINED
VEC_AVX512F_XRSHIFT(rshift, 64, 8, /* nothing */, a)
# define VINT64x8_RSHIFT_DEFINED
#endif

#ifndef VUINT32x16_RSHIFT_DEFINED
VEC_AVX512F_XRSHIFT(rshift, 32, 16, u, l)
# define VUINT32x16_RSHIFT_DEFINED
#endif

#ifndef VUINT64x8_RSHIFT_DEFINED
VEC_AVX512F_XRSHIFT(rshift, 64, 8, u, l)
# define VUINT64x8_RSHIFT_DEFINED
#endif

/* ------------------------------------------------------------------------ */

#define VEC_AVX512F_SPLAT(sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_splat(vec_##sign##int##size x) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.avx512f = _mm512_set1_epi##bits(x); \
		return vec; \
	}

#define VEC_AVX512F_LOAD_ALIGNED(sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_load_aligned(const vec_##sign##int##bits in[size]) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.avx512f = _mm512_load_si512((const __m512i *)in); \
		return vec; \
	}

#define VEC_AVX512F_LOAD(sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_load(const vec_##sign##int##bits in[size]) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.avx512f = _mm512_loadu_si512((const __m512i *)in); \
		return vec; \
	}

#define VEC_AVX512F_STORE_ALIGNED(sign, bits, size) \
	VEC_FUNC_IMPL void v##sign##int##bits##x##size##_store_aligned(v##sign##int##bits##x##size vec, vec_##sign##int##bits out[size]) \
	{ \
		_mm512_store_si512((__m512i *)out, vec.avx512f); \
	}

#define VEC_AVX512F_STORE(sign, bits, size) \
	VEC_FUNC_IMPL void v##sign##int##bits##x##size##_store(v##sign##int##bits##x##size vec, vec_##sign##int##bits out[size]) \
	{ \
		_mm512_storeu_si512((__m512i *)out, vec.avx512f); \
	}

#define VEC_AVX512F_BITWISE(op, sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_##op(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		vec1.avx512f = _mm512_##op##_si512(vec1.avx512f, vec2.avx512f); \
		return vec1; \
	}

/* ------------------------------------------------------------------------ */
/* 8x64; there is very little we can do here */

#ifndef VINT8x64_SPLAT_DEFINED
VEC_AVX512F_SPLAT(, 8, 64)
# define VINT8x64_SPLAT_DEFINED
#endif

#ifndef VINT8x64_LOAD_ALIGNED_DEFINED
VEC_AVX512F_LOAD_ALIGNED(, 8, 64)
# define VINT8x64_LOAD_ALIGNED_DEFINED
#endif

#ifndef VINT8x64_LOAD_DEFINED
VEC_AVX512F_LOAD(, 8, 64)
# define VINT8x64_LOAD_DEFINED
#endif

#ifndef VINT8x64_STORE_ALIGNED_DEFINED
VEC_AVX512F_STORE_ALIGNED(, 8, 64)
# define VINT8x64_STORE_ALIGNED_DEFINED
#endif

#ifndef VINT8x64_STORE_DEFINED
VEC_AVX512F_STORE(, 8, 64)
# define VINT8x64_STORE_DEFINED
#endif

#ifndef VINT8x64_AND_DEFINED
VEC_AVX512F_BITWISE(and, , 8, 64)
# define VINT8x64_AND_DEFINED
#endif

#ifndef VINT8x64_OR_DEFINED
VEC_AVX512F_BITWISE(or, , 8, 64)
# define VINT8x64_OR_DEFINED
#endif

#ifndef VINT8x64_XOR_DEFINED
VEC_AVX512F_BITWISE(xor, , 8, 64)
# define VINT8x64_XOR_DEFINED
#endif

/* ---- unsigned */

#ifndef VUINT8x64_SPLAT_DEFINED
VEC_AVX512F_SPLAT(u, 8, 64)
# define VUINT8x64_SPLAT_DEFINED
#endif

#ifndef VUINT8x64_LOAD_ALIGNED_DEFINED
VEC_AVX512F_LOAD_ALIGNED(u, 8, 64)
# define VUINT8x64_LOAD_ALIGNED_DEFINED
#endif

#ifndef VUINT8x64_LOAD_DEFINED
VEC_AVX512F_LOAD(u, 8, 64)
# define VUINT8x64_LOAD_DEFINED
#endif

#ifndef VUINT8x64_STORE_ALIGNED_DEFINED
VEC_AVX512F_STORE_ALIGNED(u, 8, 64)
# define VUINT8x64_STORE_ALIGNED_DEFINED
#endif

#ifndef VUINT8x64_STORE_DEFINED
VEC_AVX512F_STORE(u, 8, 64)
# define VUINT8x64_STORE_DEFINED
#endif

#ifndef VUINT8x64_AND_DEFINED
VEC_AVX512F_BITWISE(and, u, 8, 64)
# define VUINT8x64_AND_DEFINED
#endif

#ifndef VUINT8x64_OR_DEFINED
VEC_AVX512F_BITWISE(or, u, 8, 64)
# define VUINT8x64_OR_DEFINED
#endif

#ifndef VUINT8x64_XOR_DEFINED
VEC_AVX512F_BITWISE(xor, u, 8, 64)
# define VUINT8x64_XOR_DEFINED
#endif

/* ------------------------------------------------------------------------ */
/* 16x32; there is very little we can do here */

#ifndef VINT16x32_SPLAT_DEFINED
VEC_AVX512F_SPLAT(, 16, 32)
# define VINT16x32_SPLAT_DEFINED
#endif

#ifndef VINT16x32_LOAD_ALIGNED_DEFINED
VEC_AVX512F_LOAD_ALIGNED(, 16, 32)
# define VINT16x32_LOAD_ALIGNED_DEFINED
#endif

#ifndef VINT16x32_LOAD_DEFINED
VEC_AVX512F_LOAD(, 16, 32)
# define VINT16x32_LOAD_DEFINED
#endif

#ifndef VINT16x32_STORE_ALIGNED_DEFINED
VEC_AVX512F_STORE_ALIGNED(, 16, 32)
# define VINT16x32_STORE_ALIGNED_DEFINED
#endif

#ifndef VINT16x32_STORE_DEFINED
VEC_AVX512F_STORE(, 16, 32)
# define VINT16x32_STORE_DEFINED
#endif

#ifndef VINT16x32_AND_DEFINED
VEC_AVX512F_BITWISE(and, , 16, 32)
# define VINT16x32_AND_DEFINED
#endif

#ifndef VINT16x32_OR_DEFINED
VEC_AVX512F_BITWISE(or, , 16, 32)
# define VINT16x32_OR_DEFINED
#endif

#ifndef VINT16x32_XOR_DEFINED
VEC_AVX512F_BITWISE(xor, , 16, 32)
# define VINT16x32_XOR_DEFINED
#endif

/* ---- unsigned */

#ifndef VUINT16x32_SPLAT_DEFINED
VEC_AVX512F_SPLAT(u, 16, 32)
# define VUINT16x32_SPLAT_DEFINED
#endif

#ifndef VUINT16x32_LOAD_ALIGNED_DEFINED
VEC_AVX512F_LOAD_ALIGNED(u, 16, 32)
# define VUINT16x32_LOAD_ALIGNED_DEFINED
#endif

#ifndef VUINT16x32_LOAD_DEFINED
VEC_AVX512F_LOAD(u, 16, 32)
# define VUINT16x32_LOAD_DEFINED
#endif

#ifndef VUINT16x32_STORE_ALIGNED_DEFINED
VEC_AVX512F_STORE_ALIGNED(u, 16, 32)
# define VUINT16x32_STORE_ALIGNED_DEFINED
#endif

#ifndef VUINT16x32_STORE_DEFINED
VEC_AVX512F_STORE(u, 16, 32)
# define VUINT16x32_STORE_DEFINED
#endif

#ifndef VUINT16x32_AND_DEFINED
VEC_AVX512F_BITWISE(and, u, 16, 32)
# define VUINT16x32_AND_DEFINED
#endif

#ifndef VUINT16x32_OR_DEFINED
VEC_AVX512F_BITWISE(or, u, 16, 32)
# define VUINT16x32_OR_DEFINED
#endif

#ifndef VUINT16x32_XOR_DEFINED
VEC_AVX512F_BITWISE(xor, u, 16, 32)
# define VUINT16x32_XOR_DEFINED
#endif

/* ------------------------------------------------------------------------ */

#ifndef VINT32x16_SPLAT_DEFINED
VEC_AVX512F_SPLAT(, 32, 16)
# define VINT32x16_SPLAT_DEFINED
#endif

#ifndef VINT32x16_LOAD_ALIGNED_DEFINED
VEC_AVX512F_LOAD_ALIGNED(, 32, 16)
# define VINT32x16_LOAD_ALIGNED_DEFINED
#endif

#ifndef VINT32x16_LOAD_DEFINED
VEC_AVX512F_LOAD(, 32, 16)
# define VINT32x16_LOAD_DEFINED
#endif

#ifndef VINT32x16_STORE_ALIGNED_DEFINED
VEC_AVX512F_STORE_ALIGNED(, 32, 16)
# define VINT32x16_STORE_ALIGNED_DEFINED
#endif

#ifndef VINT32x16_STORE_DEFINED
VEC_AVX512F_STORE(, 32, 16)
# define VINT32x16_STORE_DEFINED
#endif

#ifndef VINT32x16_ADD_DEFINED
VEC_AVX512F_OPERATION_EX_EX(add, add, , 32, 16, , i)
# define VINT32x16_ADD_DEFINED
#endif

#ifndef VINT32x16_SUB_DEFINED
VEC_AVX512F_OPERATION_EX_EX(sub, sub, , 32, 16, , i)
# define VINT32x16_SUB_DEFINED
#endif

#ifndef VINT32x16_MUL_DEFINED
VEC_AVX512F_OPERATION_EX_EX(mul, mullo, , 32, 16, , i)
# define VINT32x16_MUL_DEFINED
#endif

#ifndef VINT32x16_AND_DEFINED
VEC_AVX512F_BITWISE(and, , 32, 16)
# define VINT32x16_AND_DEFINED
#endif

#ifndef VINT32x16_OR_DEFINED
VEC_AVX512F_BITWISE(or, , 32, 16)
# define VINT32x16_OR_DEFINED
#endif

#ifndef VINT32x16_XOR_DEFINED
VEC_AVX512F_BITWISE(xor, , 32, 16)
# define VINT32x16_XOR_DEFINED
#endif

#ifndef VUINT32x16_SPLAT_DEFINED
VEC_AVX512F_SPLAT(u, 32, 16)
# define VUINT32x16_SPLAT_DEFINED
#endif

#ifndef VUINT32x16_LOAD_ALIGNED_DEFINED
VEC_AVX512F_LOAD_ALIGNED(u, 32, 16)
# define VUINT32x16_LOAD_ALIGNED_DEFINED
#endif

#ifndef VUINT32x16_LOAD_DEFINED
VEC_AVX512F_LOAD(u, 32, 16)
# define VUINT32x16_LOAD_DEFINED
#endif

#ifndef VUINT32x16_STORE_ALIGNED_DEFINED
VEC_AVX512F_STORE_ALIGNED(u, 32, 16)
# define VUINT32x16_STORE_ALIGNED_DEFINED
#endif

#ifndef VUINT32x16_STORE_DEFINED
VEC_AVX512F_STORE(u, 32, 16)
# define VUINT32x16_STORE_DEFINED
#endif

#ifndef VUINT32x16_ADD_DEFINED
VEC_AVX512F_OPERATION_EX_EX(add, add, u, 32, 16, u, i)
# define VUINT32x16_ADD_DEFINED
#endif

#ifndef VUINT32x16_SUB_DEFINED
VEC_AVX512F_OPERATION_EX_EX(sub, sub, u, 32, 16, u, i)
# define VUINT32x16_SUB_DEFINED
#endif

#ifndef VUINT32x16_MUL_DEFINED
VEC_AVX512F_OPERATION_EX_EX(mul, mullo, u, 32, 16, u, i)
# define VUINT32x16_MUL_DEFINED
#endif

#ifndef VUINT32x16_AND_DEFINED
VEC_AVX512F_BITWISE(and, u, 32, 16)
# define VUINT32x16_AND_DEFINED
#endif

#ifndef VUINT32x16_OR_DEFINED
VEC_AVX512F_BITWISE(or, u, 32, 16)
# define VUINT32x16_OR_DEFINED
#endif

#ifndef VUINT32x16_XOR_DEFINED
VEC_AVX512F_BITWISE(xor, u, 32, 16)
# define VUINT32x16_XOR_DEFINED
#endif

#ifndef VINT64x8_SPLAT_DEFINED
VEC_AVX512F_SPLAT(, 64, 8)
# define VINT64x8_SPLAT_DEFINED
#endif

#ifndef VINT64x8_LOAD_ALIGNED_DEFINED
VEC_AVX512F_LOAD_ALIGNED(, 64, 8)
# define VINT64x8_LOAD_ALIGNED_DEFINED
#endif

#ifndef VINT64x8_LOAD_DEFINED
VEC_AVX512F_LOAD(, 64, 8)
# define VINT64x8_LOAD_DEFINED
#endif

#ifndef VINT64x8_STORE_ALIGNED_DEFINED
VEC_AVX512F_STORE_ALIGNED(, 64, 8)
# define VINT64x8_STORE_ALIGNED_DEFINED
#endif

#ifndef VINT64x8_STORE_DEFINED
VEC_AVX512F_STORE(, 64, 8)
# define VINT64x8_STORE_DEFINED
#endif

#ifndef VINT64x8_ADD_DEFINED
VEC_AVX512F_OPERATION_EX_EX(add, add, , 64, 8, , i)
# define VINT64x8_ADD_DEFINED
#endif

#ifndef VINT64x8_SUB_DEFINED
VEC_AVX512F_OPERATION_EX_EX(sub, sub, , 64, 8, , i)
# define VINT64x8_SUB_DEFINED
#endif

#ifndef VINT64x8_MUL_DEFINED
VEC_AVX512F_OPERATION_EX_EX(mul, mullox, , 64, 8, , i)
# define VINT64x8_MUL_DEFINED
#endif

#ifndef VINT64x8_AND_DEFINED
VEC_AVX512F_BITWISE(and, , 64, 8)
# define VINT64x8_AND_DEFINED
#endif

#ifndef VINT64x8_OR_DEFINED
VEC_AVX512F_BITWISE(or, , 64, 8)
# define VINT64x8_OR_DEFINED
#endif

#ifndef VINT64x8_XOR_DEFINED
VEC_AVX512F_BITWISE(xor, , 64, 8)
# define VINT64x8_XOR_DEFINED
#endif

#ifndef VUINT64x8_SPLAT_DEFINED
VEC_AVX512F_SPLAT(u, 64, 8)
# define VUINT64x8_SPLAT_DEFINED
#endif

#ifndef VUINT64x8_LOAD_ALIGNED_DEFINED
VEC_AVX512F_LOAD_ALIGNED(u, 64, 8)
# define VUINT64x8_LOAD_ALIGNED_DEFINED
#endif

#ifndef VUINT64x8_LOAD_DEFINED
VEC_AVX512F_LOAD(u, 64, 8)
# define VUINT64x8_LOAD_DEFINED
#endif

#ifndef VUINT64x8_STORE_ALIGNED_DEFINED
VEC_AVX512F_STORE_ALIGNED(u, 64, 8)
# define VUINT64x8_STORE_ALIGNED_DEFINED
#endif

#ifndef VUINT64x8_STORE_DEFINED
VEC_AVX512F_STORE(u, 64, 8)
# define VUINT64x8_STORE_DEFINED
#endif

#ifndef VUINT64x8_ADD_DEFINED
VEC_AVX512F_OPERATION_EX_EX(add, add, u, 64, 8, u, i)
# define VUINT64x8_ADD_DEFINED
#endif

#ifndef VUINT64x8_SUB_DEFINED
VEC_AVX512F_OPERATION_EX_EX(sub, sub, u, 64, 8, u, i)
# define VUINT64x8_SUB_DEFINED
#endif

#ifndef VUINT64x8_MUL_DEFINED
VEC_AVX512F_OPERATION_EX_EX(mul, mullox, u, 64, 8, u, i)
# define VUINT64x8_MUL_DEFINED
#endif

#ifndef VUINT64x8_AND_DEFINED
VEC_AVX512F_BITWISE(and, u, 64, 8)
# define VUINT64x8_AND_DEFINED
#endif

#ifndef VUINT64x8_OR_DEFINED
VEC_AVX512F_BITWISE(or, u, 64, 8)
# define VUINT64x8_OR_DEFINED
#endif

#ifndef VUINT64x8_XOR_DEFINED
VEC_AVX512F_BITWISE(xor, u, 64, 8)
# define VUINT64x8_XOR_DEFINED
#endif

#endif /* VEC_IMPL_X86_AVX512F_H_ */
