/**
 * vec - a tiny SIMD vector library in C99
 * 
 * Copyright (c) 2024 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

#include "vec/vec.h"
#include "vec/impl/x86/mmx.h"

#include <mmintrin.h>
#include <string.h>

/* ------------------------------------------------------------------------ */

#define VEC_MMX_MUL_8x8(sign) /* nothing */
#define VEC_MMX_MUL_16x4(sign) \
	VEC_FUNC_IMPL v##sign##int16x4 v##sign##int16x4_mmx_mul(v##sign##int16x4 vec1, v##sign##int16x4 vec2) \
	{ \
		union v##sign##int16x4_impl_data *vec1d = (union v##sign##int16x4_impl_data *)&vec1; \
		union vuint16x4_impl_data *vec2d = (union vuint16x4_impl_data *)&vec2; \
	\
		vec1d->mmx = _mm_mullo_pi16(vec1d->mmx, vec2d->mmx); \
		return vec1d->vec; \
	}
#define VEC_MMX_MUL_32x2(sign) /* nothing */

#define VEC_MMX_STRUCT_MUL_8x8(sign) NULL
#define VEC_MMX_STRUCT_MUL_16x4(sign) v##sign##int16x4_mmx_mul
#define VEC_MMX_STRUCT_MUL_32x8(sign) NULL

/* ------------------------------------------------------------------------ */
/* comparison */

/* helper funcs */
#define VEC_xMMX_CMP(name, op, sign, bits, size, first, second, VARS, TRANS1, TRANS2) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_##name(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
		VARS \
	\
		TRANS1 \
	\
		vec1d->mmx = _mm_##op##_pi##bits(vec##first##d->mmx, vec##second##d->mmx); \
	\
		TRANS2 \
	\
		return vec1d->vec; \
	}

#define VEC_MMX_CMP(name, op, bits, size, first, second) \
	VEC_xMMX_CMP(name, op, /* nothing */, bits, size, first, second, /* nothing */, /* nothing */, /* nothing */)

#define VEC_uMMX_CMP(name, op, bits, size, first, second) \
	VEC_xMMX_CMP(name, op, u, bits, size, first, second, \
		__m64 xor_val = _mm_set1_pi##bits(1u << (bits - 1)); \
	, { \
		vec1d->mmx = _mm_xor_si64(vec1d->mmx, xor_val); \
		vec2d->mmx = _mm_xor_si64(vec2d->mmx, xor_val); \
	}, \
	{ \
		/* nothing */ \
	})

#define VEC_MMX_CMPEQ(sign, bits, size) VEC_xMMX_CMP(cmpeq, cmpeq, sign, bits, size, 1, 2, , ,)
#define VEC_MMX_CMPLT(sign, bits, size) VEC_##sign##MMX_CMP(cmplt, cmpgt, bits, size, 2, 1)
#define VEC_MMX_CMPGT(sign, bits, size) VEC_##sign##MMX_CMP(cmpgt, cmpgt, bits, size, 1, 2)

/* ------------------------------------------------------------------------ */

#define VEC_MMX_DEFINE_OPERATIONS_SIGN(sign, bits, size) \
	union v##sign##int##bits##x##size##_impl_data { \
		v##sign##int##bits##x##size vec; \
		__m64 mmx; \
	}; \
	\
	VEC_STATIC_ASSERT(VEC_ALIGNOF(__m64) <= VEC_ALIGNOF(v##sign##int##bits##x##size), "vec: v" #sign "int" #bits "x" #size " alignment needs to be expanded to fit intrinsic type size"); \
	VEC_STATIC_ASSERT(sizeof(__m64) <= sizeof(v##sign##int##bits##x##size), "vec: v" #sign "int" #bits "x" #size " needs to be expanded to fit intrinsic type size"); \
	\
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_load_aligned(const vec_##sign##int##bits in[size]) \
	{ \
		v##sign##int##bits##x##size vec; \
		memcpy(&vec, in, sizeof(vec)); \
		return vec; \
	} \
	\
	VEC_FUNC_IMPL void v##sign##int##bits##x##size##_mmx_store_aligned(v##sign##int##bits##x##size vec, vec_##sign##int##bits out[size]) \
	{ \
		memcpy(out, &vec, sizeof(vec)); \
	} \
	\
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_add(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->mmx = _mm_add_pi##bits(vec1d->mmx, vec2d->mmx); \
	\
		return vec1d->vec; \
	} \
	\
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_sub(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->mmx = _mm_sub_pi##bits(vec1d->mmx, vec2d->mmx); \
	\
		return vec1d->vec; \
	} \
	\
	VEC_MMX_MUL_##bits##x##size(sign) \
	\
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_and(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->mmx = _mm_and_si64(vec1d->mmx, vec2d->mmx); \
	\
		return vec1d->vec; \
	} \
	\
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_or(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->mmx = _mm_or_si64(vec1d->mmx, vec2d->mmx); \
	\
		return vec1d->vec; \
	} \
	\
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_xor(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->mmx = _mm_xor_si64(vec1d->mmx, vec2d->mmx); \
	\
		return vec1d->vec; \
	} \
	\
	VEC_MMX_CMPEQ(sign, bits, size) \
	VEC_MMX_CMPLT(sign, bits, size) \
	VEC_MMX_CMPGT(sign, bits, size) \
	\
	const v##sign##int##bits##x##size##_impl v##sign##int##bits##x##size##_impl_mmx = { \
		.load_aligned = v##sign##int##bits##x##size##_mmx_load_aligned, \
		.load = v##sign##int##bits##x##size##_mmx_load_aligned, \
		.store_aligned = v##sign##int##bits##x##size##_mmx_store_aligned, \
		.store = v##sign##int##bits##x##size##_mmx_store_aligned, \
		.add = v##sign##int##bits##x##size##_mmx_add, \
		.sub = v##sign##int##bits##x##size##_mmx_sub, \
		.mul = VEC_MMX_STRUCT_MUL_8x8(sign), \
		.band = v##sign##int##bits##x##size##_mmx_and, \
		.bor = v##sign##int##bits##x##size##_mmx_or, \
		.bxor = v##sign##int##bits##x##size##_mmx_xor, \
		.cmpeq = v##sign##int##bits##x##size##_mmx_cmpeq, \
	};

#define VEC_MMX_DEFINE_OPERATIONS(bits, size) \
	VEC_MMX_DEFINE_OPERATIONS_SIGN(u, bits, size) \
	VEC_MMX_DEFINE_OPERATIONS_SIGN( , bits, size)

VEC_MMX_DEFINE_OPERATIONS(8, 8)
VEC_MMX_DEFINE_OPERATIONS(16, 4)
VEC_MMX_DEFINE_OPERATIONS(32, 2)
