/**
 * vec - a tiny SIMD vector library in C99
 * 
 * Copyright (c) 2024 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

#include "vec/vec.h"
#include "vec/impl/x86/mmx.h"
#include "vec/impl/generic.h"

#include <mmintrin.h>
#include <string.h>

#define VEC_MMX_OPERATION_8x8(op, sign) \
	do { \
		/* unpack and multiply */ \
		union v##sign##int8x8_impl_data *vec1d = (union v##sign##int8x8_impl_data *)&vec1; \
		union v##sign##int8x8_impl_data *vec2d = (union v##sign##int8x8_impl_data *)&vec2; \
	\
		__m64 dst_even = _mm_##op##_pi16(vec1d->mmx, vec2d->mmx); \
		__m64 dst_odd = _mm_##op##_pi16(_mm_srli_pi16(vec1d->mmx, 8), _mm_srli_pi16(vec2d->mmx, 8)); \
	\
		/* repack */ \
		vec1d->mmx = _mm_or_si64( \
			_mm_slli_pi16(dst_odd, 8), \
			_mm_srli_pi16(_mm_slli_pi16(dst_even, 8), 8) \
		); \
		return vec1d->vec; \
	} while (0)

// shared between MMX variations
#define VEC_MMX_MUL_8x8(sign) \
	VEC_MMX_OPERATION_8x8(mullo, sign)

#define VEC_MMX_MUL_16x4(sign) \
	do { \
		union v##sign##int16x4_impl_data *vec1d = (union v##sign##int16x4_impl_data *)&vec1; \
		union vuint16x4_impl_data *vec2d = (union vuint16x4_impl_data *)&vec2; \
	\
		vec1d->mmx = _mm_mullo_pi16(vec1d->mmx, vec2d->mmx); \
		return vec1d->vec; \
	} while (0)

#define VEC_MMX_MUL_32x2(sign) \
	/* TODO implement this for real */ \
	do { \
		return v##sign##int32x2_generic_mul(vec1, vec2); \
	} while (0)

#define VEC_MMX_DEFINE_OPERATIONS_SIGN(sign, bits, size) \
	union v##sign##int##bits##x##size##_impl_data { \
		v##sign##int##bits##x##size vec; \
		__m64 mmx; \
	}; \
	\
	VEC_STATIC_ASSERT(VEC_ALIGNOF(__m64) <= VEC_ALIGNOF(v##sign##int##bits##x##size), "vec: v" #sign "int" #bits "x" #size " alignment needs to be expanded to fit intrinsic type size"); \
	VEC_STATIC_ASSERT(sizeof(__m64) <= sizeof(v##sign##int##bits##x##size), "vec: v" #sign "int" #bits "x" #size " needs to be expanded to fit intrinsic type size"); \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_load_aligned(const vec_##sign##int##bits in[size]) \
	{ \
		v##sign##int##bits##x##size vec; \
		memcpy(&vec, in, sizeof(vec)); \
		return vec; \
	} \
	\
	static void v##sign##int##bits##x##size##_mmx_store_aligned(v##sign##int##bits##x##size vec, vec_##sign##int##bits out[size]) \
	{ \
		memcpy(out, &vec, sizeof(vec)); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_add(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->mmx = _mm_add_pi##bits(vec1d->mmx, vec2d->mmx); \
	\
		return vec1d->vec; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_sub(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->mmx = _mm_sub_pi##bits(vec1d->mmx, vec2d->mmx); \
	\
		return vec1d->vec; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_mul(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		VEC_MMX_MUL_##bits##x##size(sign); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_and(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->mmx = _mm_and_si64(vec1d->mmx, vec2d->mmx); \
	\
		return vec1d->vec; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_or(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->mmx = _mm_or_si64(vec1d->mmx, vec2d->mmx); \
	\
		return vec1d->vec; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_xor(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->mmx = _mm_xor_si64(vec1d->mmx, vec2d->mmx); \
	\
		return vec1d->vec; \
	} \
	\
	const v##sign##int##bits##x##size##_impl v##sign##int##bits##x##size##_impl_mmx = { \
		v##sign##int##bits##x##size##_generic_splat, \
		v##sign##int##bits##x##size##_mmx_load_aligned, \
		v##sign##int##bits##x##size##_mmx_load_aligned, \
		v##sign##int##bits##x##size##_mmx_store_aligned, \
		v##sign##int##bits##x##size##_mmx_store_aligned, \
		v##sign##int##bits##x##size##_mmx_add, \
		v##sign##int##bits##x##size##_mmx_sub, \
		v##sign##int##bits##x##size##_mmx_mul, \
		v##sign##int##bits##x##size##_generic_div, \
		v##sign##int##bits##x##size##_generic_avg, \
		v##sign##int##bits##x##size##_mmx_and, \
		v##sign##int##bits##x##size##_mmx_or, \
		v##sign##int##bits##x##size##_mmx_xor, \
		v##sign##int##bits##x##size##_generic_not, \
		v##sign##int##bits##x##size##_generic_lshift, \
		v##sign##int##bits##x##size##_generic_rshift, \
		v##sign##int##bits##x##size##_generic_lrshift, \
		v##sign##int##bits##x##size##_generic_cmplt, \
		v##sign##int##bits##x##size##_generic_cmple, \
		v##sign##int##bits##x##size##_generic_cmpeq, \
		v##sign##int##bits##x##size##_generic_cmpge, \
		v##sign##int##bits##x##size##_generic_cmpgt, \
	};

#define VEC_MMX_DEFINE_OPERATIONS(bits, size) \
	VEC_MMX_DEFINE_OPERATIONS_SIGN(u, bits, size) \
	VEC_MMX_DEFINE_OPERATIONS_SIGN( , bits, size)

VEC_MMX_DEFINE_OPERATIONS(8, 8)
VEC_MMX_DEFINE_OPERATIONS(16, 4)
VEC_MMX_DEFINE_OPERATIONS(32, 2)
