#include "vec/vec.h"
#include "vec/mem.h"

#include <stdio.h>
#include <string.h>
#include <inttypes.h>
#include <time.h>

#define ARRAY_SIZE(x) (sizeof(x)/sizeof((x)[0]))

static const int8_t testval8[] = {
	INT8_C(-80), INT8_C(-3), INT8_C(25), INT8_C(0x7F),
	INT8_C(-42), INT8_C(27), INT8_C(24), INT8_C(0x40),
};

static const uint8_t testvalu8[] = {
	UINT8_C(0x00), UINT8_C(0xFF), UINT8_C(0xFE), UINT8_C(0x7F),
	UINT8_C(0xC0), UINT8_C(0x80), UINT8_C(0x20), UINT8_C(0x50),
};

static const int16_t testval16[] = {
	INT16_C(-8000), INT16_C(-30), INT16_MAX, INT16_C(0x4000),
	INT16_C(-42),   INT16_C(250), INT16_MIN, INT16_C(0x500),
};

static const uint16_t testvalu16[] = {
	UINT16_C(0x0000), UINT16_C(0xFFFF), UINT16_C(0xFEA), UINT16_C(0x7FF),
	UINT16_C(0x7FFF), UINT16_C(0x8000), UINT16_C(0x20B), UINT16_C(0x50C),
};

static const int32_t testval32[] = {
	INT32_C(-1000000),   INT32_C(-3), INT32_C(0x00000000), INT32_C(0xFFFFFFFF),
	INT32_C(     -42),   INT32_C(27), INT32_C(0xABCDEF03), INT32_C(0x00000FFF),
	INT32_C(0xFFFFFFFF), INT32_C( 0), INT32_C(0xFFFFFFFE), INT32_C(         1),
};

static const uint32_t testvalu32[] = {
	UINT32_C(0x00000000), UINT32_C(0xDEADBEEF), UINT32_C(42), UINT32_C(0x12340000),
	UINT32_C(0xFFFFFFFF), UINT32_C(0xFEDCBA98), UINT32_C(17), UINT32_C(0x00012345),
	UINT32_C(0xFFFFFFFF), UINT32_C(0xFFFFFFFE), UINT32_C( 0), UINT32_C(         1),
};

static const float testvalf32[] = {
	1.0f, -3.33f, -4096.0f, 1234.0f,
	90.0f, -12.0f, 60.0f, 10224.0f,
};

static const int64_t testval64[] = {
	INT64_MAX, INT64_C(-3),     INT64_C(0x00000000),   INT64_C(0xFFFFFFFFF),
	INT64_MIN, INT64_C(645366), INT64_C(0x12345ABCDE), INT64_C(0xF00000FFF),
};

static const uint64_t testvalu64[] = {
	UINT64_MAX,     UINT64_C(0x44354365), UINT64_C(0x00000000),   UINT64_C(0xFFFFFFFFF),
	UINT64_C(0xff), UINT64_C(645366),     UINT64_C(0x12345ABCDE), UINT64_C(0xF00000FFF),
};

static const double testvalf64[] = {
	2345734.0, 12498.0, 12.0, -12312.0,
	-5.0, 12.234, 3.1415, 2.71828,
};

#define VTEST(shorttype, type, ctype, bits, size) \
	static inline v##type##bits##x##size vtest##shorttype##bits##x##size(const size_t start) \
	{ \
		V##ctype##bits##x##size##_ALIGNED_ARRAY(x); \
		for (size_t i = 0; i < size; i++) \
			x[i] = testval##shorttype##bits[(start + i) % ARRAY_SIZE(testval##shorttype##bits)]; \
		return v##type##bits##x##size##_load_aligned(x); \
	}

#define VPRINT(type, ctype, print, bits, size) \
	static inline void print_v##type##bits##x##size(FILE *file, v##type##bits##x##size vec) \
	{ \
		fputs("vector: ", file); \
	\
		V##ctype##bits##x##size##_ALIGNED_ARRAY(v); \
	\
		v##type##bits##x##size##_store_aligned(vec, v); \
	\
		fprintf(file, "%" print, v[0]); \
	\
		for (int i = 1; i < size; i++) \
			fprintf(file, ", %" print, v[i]); \
	\
		fputs("\n", file); \
	}

#define DEF_VEC_TEST_FUNCS(bits, size) \
	VTEST(, int, INT, bits, size)     VTEST(u, uint, UINT, bits, size) \
	VPRINT(int, INT, PRI##d##bits, bits, size) VPRINT(uint, UINT, PRI##u##bits, bits, size)

#define DEF_VEC_TEST_FUNC_FLOAT(bits, size) \
	VTEST(f, f, F, bits, size) \
	VPRINT(f, F, "f", bits, size)

DEF_VEC_TEST_FUNCS(8, 2)

DEF_VEC_TEST_FUNCS(8, 4)
DEF_VEC_TEST_FUNCS(16, 2)

DEF_VEC_TEST_FUNCS(8, 8)
DEF_VEC_TEST_FUNCS(16, 4)
DEF_VEC_TEST_FUNCS(32, 2)

DEF_VEC_TEST_FUNCS(8, 16)
DEF_VEC_TEST_FUNCS(16, 8)
DEF_VEC_TEST_FUNCS(32, 4)
DEF_VEC_TEST_FUNCS(64, 2)

DEF_VEC_TEST_FUNCS(8, 32)
DEF_VEC_TEST_FUNCS(16, 16)
DEF_VEC_TEST_FUNCS(32, 8)
DEF_VEC_TEST_FUNCS(64, 4)

DEF_VEC_TEST_FUNCS(8, 64)
DEF_VEC_TEST_FUNCS(16, 32)
DEF_VEC_TEST_FUNCS(32, 16)
DEF_VEC_TEST_FUNCS(64, 8)

DEF_VEC_TEST_FUNC_FLOAT(32, 2)
DEF_VEC_TEST_FUNC_FLOAT(32, 4)
DEF_VEC_TEST_FUNC_FLOAT(32, 8)
DEF_VEC_TEST_FUNC_FLOAT(32, 16)

DEF_VEC_TEST_FUNC_FLOAT(64, 2)
DEF_VEC_TEST_FUNC_FLOAT(64, 4)
DEF_VEC_TEST_FUNC_FLOAT(64, 8)

#undef DEF_VEC_TEST_FUNCS
#undef VPRINT
#undef VTEST

// ------------------------------------------------------------

#include "test_align.h"
#include "test_arith.h"
#include "test_compare.h"
#include "test_shift.h"
#include "test_benchmark.h"

// ------------------------------------------------------------

int main(void)
{
	int ret = 0;

	srand(time(NULL));

	ret |= test_align();
	ret |= test_arith();
	ret |= test_compare();
	ret |= test_shift();

	test_benchmark();

	return ret;
}
