/**
 * vec - a tiny SIMD vector library in C99
 * 
 * Copyright (c) 2024-2025 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

#ifndef VEC_MEM_H_
#define VEC_MEM_H_

#include "defs.h"

#define VEC_MALLOC_ALIGNMENT (64)

VEC_STATIC_ASSERT(!(VEC_MALLOC_ALIGNMENT & (VEC_MALLOC_ALIGNMENT - 1))
	&& (VEC_MALLOC_ALIGNMENT > 0),
	"VEC_MALLOC_ALIGNMENT must be a power of two");

typedef unsigned char vec_alignment_type;

VEC_STATIC_ASSERT(VEC_MALLOC_ALIGNMENT <= VEC_MAX_OF_TYPE(vec_alignment_type),
	"VEC_MALLOC_ALIGNMENT cannot fit in the pointer alignment data");

#define VEC_MALLOC_ADDITIONAL_SIZE (sizeof(vec_alignment_type) + (VEC_MALLOC_ALIGNMENT - 1))
#define VEC_MALLOC_MAX_SIZE (SIZE_MAX - VEC_MALLOC_ADDITIONAL_SIZE)

VEC_FUNC_IMPL void *vec_internal_align_ptr_(void *q)
{
	vec_alignment_type diff;

	diff = (((uintptr_t)q + (VEC_MALLOC_ALIGNMENT - 1)) & ~(VEC_MALLOC_ALIGNMENT - 1)) - (uintptr_t)q;
	q = (char *)q + diff;

	memcpy((char *)q - sizeof(diff), &diff, sizeof(diff));

	return q;
}

/* reverses vec_align_ptr */
VEC_FUNC_IMPL void *vec_internal_unalign_ptr_(void *q)
{
	vec_alignment_type diff;

	memcpy(&diff, (char *)q - sizeof(diff), sizeof(diff));
	q = (char *)q - diff;

	return q;
}

VEC_FUNC_IMPL void *vec_malloc(size_t size)
{
	void *q;

	if (size > VEC_MALLOC_MAX_SIZE)
		return NULL;

	/* allocate space for the diff (we have to do this,
	 * for realloc has no way of knowing the original ptr) */
	q = malloc(size + VEC_MALLOC_ADDITIONAL_SIZE);
	if (!q)
		return NULL;

	return vec_internal_align_ptr_(q);
}

VEC_FUNC_IMPL void *vec_calloc(size_t count, size_t nmemb)
{
	size_t size;
	void *q;

	size = count * nmemb;
	if ((size && size / count != nmemb)
		|| size > VEC_MALLOC_MAX_SIZE)
		return NULL; /* nope */

	q = calloc(size + VEC_MALLOC_ADDITIONAL_SIZE, 1);
	if (!q)
		return NULL;

	return vec_internal_align_ptr_(q);
}

VEC_FUNC_IMPL void *vec_realloc(void *ptr, size_t newsize)
{
	void *q;

	if (!ptr)
		return vec_malloc(newsize);

	if (newsize > VEC_MALLOC_MAX_SIZE)
		return NULL;

	q = realloc(vec_internal_unalign_ptr_(ptr), VEC_MALLOC_ADDITIONAL_SIZE);
	if (!q)
		return NULL;

	return vec_internal_align_ptr_(q);
}

VEC_FUNC_IMPL void vec_free(void *ptr)
{
	if (ptr)
		free(vec_internal_unalign_ptr_(ptr));
}

#endif /* VEC_MEM_H_ */
