/**
 * vec - a tiny SIMD vector library in plain C99
 * 
 * Copyright (c) 2024 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

/* Altivec vector support. */

#ifndef VEC_IMPL_PPC_ALTIVEC_H_
#define VEC_IMPL_PPC_ALTIVEC_H_

#include <stdint.h>
#include <string.h>

#include <altivec.h>

#define VEC_ALTIVEC_ALIGNMENT 16

/* GCC 4.2.1 on Mac OS X doesn't have these for some reason */
#ifdef vec_mul
# define VEC_ALTIVEC_DEFINE_MUL(sign, csign, bits, size) \
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_altivec_mul(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .altivec = vec_mul(vec1.altivec, vec2.altivec) }; \
	}
# define VEC_ALTIVEC_STRUCT_MUL(sign, csign, bits, size) \
	.mul = v##sign##int##bits##x##size##_altivec_mul,
#else
# define VEC_ALTIVEC_DEFINE_MUL(sign, csign, bits, size)
# define VEC_ALTIVEC_STRUCT_MUL(sign, csign, bits, size)
#endif

#ifdef vec_splats
# define VEC_ALTIVEC_DEFINE_SPLAT(sign, csign, bits, size) \
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_altivec_splat(sign##int##bits##_t x) \
	{ \
		return (v##sign##int##bits##x##size) { .altivec = vec_splats(x) }; \
	}
# define VEC_ALTIVEC_STRUCT_SPLAT(sign, csign, bits, size) \
	.splat = v##sign##int##bits##x##size##_altivec_splat,
#else
# define VEC_ALTIVEC_DEFINE_SPLAT(sign, csign, bits, size)
# define VEC_ALTIVEC_STRUCT_SPLAT(sign, csign, bits, size)
#endif

#define VEC_ALTIVEC_uRSHIFT vec_sr
#define VEC_ALTIVEC_RSHIFT vec_sra

#define VEC_ALTIVEC_DEFINE_uLRSHIFT(sign, csign, bits, size) \
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_altivec_lrshift(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .altivec = vec_sr(vec1.altivec, vec2.altivec) }; \
	}
#define VEC_ALTIVEC_STRUCT_uLRSHIFT(sign, csign, bits, size) \
	.lrshift = v##sign##int##bits##x##size##_altivec_lrshift,

#define VEC_ALTIVEC_DEFINE_LRSHIFT(sign, csign, bits, size)
#define VEC_ALTIVEC_STRUCT_LRSHIFT(sign, csign, bits, size)

/* Since altivec conveniently made their API super user friendly, we can just use
 * one giant macro to define literally everything */
#define VEC_DEFINE_OPERATIONS_SIGN(sign, csign, bits, size) \
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_altivec_load_aligned(const sign##int##bits##_t in[size]) \
	{ \
		return (v##sign##int##bits##x##size) { .altivec = vec_ld(0, in) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_altivec_load(const sign##int##bits##_t in[size]) \
	{ \
		return (v##sign##int##bits##x##size) { .altivec = vec_perm(vec_ld(0, in), vec_ld(16, in), vec_lvsl(0, in)) }; \
	} \
	\
	static void v##sign##int##bits##x##size##_altivec_store_aligned(v##sign##int##bits##x##size vec, sign##int##bits##_t out[size]) \
	{ \
		vec_st(vec.altivec, 0, out); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_altivec_add(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .altivec = vec_add(vec1.altivec, vec2.altivec) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_altivec_sub(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .altivec = vec_sub(vec1.altivec, vec2.altivec) }; \
	} \
	\
	VEC_ALTIVEC_DEFINE_MUL(sign, csign, bits, size) \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_altivec_lshift(v##sign##int##bits##x##size vec1, vuint##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .altivec = vec_sl(vec1.altivec, vec2.altivec) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_altivec_rshift(v##sign##int##bits##x##size vec1, vuint##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .altivec = VEC_ALTIVEC_##sign##RSHIFT(vec1.altivec, vec2.altivec) }; \
	} \
	\
	VEC_ALTIVEC_DEFINE_##sign##LRSHIFT(sign, csign, bits, size) \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_altivec_avg(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .altivec = vec_avg(vec1.altivec, vec2.altivec) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_altivec_and(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .altivec = vec_and(vec1.altivec, vec2.altivec) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_altivec_or(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .altivec = vec_or(vec1.altivec, vec2.altivec) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_altivec_xor(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .altivec = vec_xor(vec1.altivec, vec2.altivec) }; \
	} \
	\
	VEC_ALTIVEC_DEFINE_SPLAT(sign, csign, bits, size) \
	\
	static v##sign##int##bits##x##size##_impl v##sign##int##bits##x##size##_impl_altivec = { \
		.load_aligned  = v##sign##int##bits##x##size##_altivec_load_aligned, \
		.load          = v##sign##int##bits##x##size##_altivec_load, \
		.store_aligned = v##sign##int##bits##x##size##_altivec_store_aligned, \
		.add           = v##sign##int##bits##x##size##_altivec_add, \
		.sub           = v##sign##int##bits##x##size##_altivec_sub, \
		VEC_ALTIVEC_STRUCT_MUL(sign, csign, bits, size) \
		.lshift        = v##sign##int##bits##x##size##_altivec_lshift, \
		.rshift        = v##sign##int##bits##x##size##_altivec_rshift, \
		VEC_ALTIVEC_STRUCT_##sign##LRSHIFT(sign, csign, bits, size) \
		.avg           = v##sign##int##bits##x##size##_altivec_avg, \
		.and           = v##sign##int##bits##x##size##_altivec_and, \
		.or            = v##sign##int##bits##x##size##_altivec_or, \
		.xor           = v##sign##int##bits##x##size##_altivec_xor, \
		VEC_ALTIVEC_STRUCT_SPLAT(sign, csign, bits, size) \
	};

#define VEC_DEFINE_OPERATIONS(bits, size) \
	VEC_DEFINE_OPERATIONS_SIGN( ,  , bits, size) \
	VEC_DEFINE_OPERATIONS_SIGN(u, U, bits, size)

VEC_DEFINE_OPERATIONS(8, 16)
VEC_DEFINE_OPERATIONS(16, 8)
VEC_DEFINE_OPERATIONS(32, 4)
#ifdef VEC_COMPILER_HAS_ALTIVEC_VSX
VEC_DEFINE_OPERATIONS(64, 2)
#endif

#undef VEC_DEFINE_OPERATIONS
#undef VEC_DEFINE_OPERATIONS_SIGN
#undef VEC_ALTIVEC_DEFINE_MUL
#undef VEC_ALTIVEC_STRUCT_MUL
#undef VEC_ALTIVEC_DEFINE_LRSHIFT
#undef VEC_ALTIVEC_STRUCT_LRSHIFT
#undef VEC_ALTIVEC_DEFINE_uLRSHIFT
#undef VEC_ALTIVEC_STRUCT_uLRSHIFT
#undef VEC_ALTIVEC_DEFINE_SPLAT
#undef VEC_ALTIVEC_STRUCT_SPLAT
#undef VEC_ALTIVEC_uRSHIFT
#undef VEC_ALTIVEC_RSHIFT

#endif /* VEC_IMPL_PPC_ALTIVEC_H_ */
