/**
 * vec - a tiny SIMD vector library in C99
 * 
 * Copyright (c) 2024 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

#ifndef VEC_IMPL_X86_MMX_H_
#define VEC_IMPL_X86_MMX_H_

#define VEC_MMX_OPERATION_8x8(op, sign) \
	do { \
		/* unpack and multiply */ \
		__m64 dst_even = _mm_##op##_pi16(vec1.mmx, vec2.mmx); \
		__m64 dst_odd = _mm_##op##_pi16(_mm_srli_pi16(vec1.mmx, 8), _mm_srli_pi16(vec2.mmx, 8)); \
	\
		/* repack */ \
		return (v##sign##int8x8){ .mmx = _mm_or_si64( \
			_mm_slli_pi16(dst_odd, 8), \
			_mm_srli_pi16(_mm_slli_pi16(dst_even, 8), 8) \
		)}; \
	} while (0)

// shifting
#define VEC_MMX_LSHIFT_8x8(sign) \
	VEC_MMX_OPERATION_8x8(sll, sign)

#define VEC_MMX_LSHIFT_16x4(sign) \
	do { \
		return (v##sign##int16x4){ .mmx = _mm_sll_pi16(vec1.mmx, vec2.mmx) }; \
	} while (0)

#define VEC_MMX_LSHIFT_32x2(sign) \
	do { \
		return (v##sign##int32x2){ .mmx = _mm_sll_pi32(vec1.mmx, vec2.mmx) }; \
	} while (0)

#define VEC_MMX_RSHIFT_8x8(sign, aORl) \
	VEC_MMX_OPERATION_8x8(sr##aORl, sign)

#define VEC_MMX_RSHIFT_16x4(sign, aORl) \
	do { \
		return (v##sign##int16x4){ .mmx = _mm_sr##aORl##_pi16(vec1.mmx, vec2.mmx) }; \
	} while (0)

#define VEC_MMX_RSHIFT_32x2(sign, aORl) \
	do { \
		return (v##sign##int32x2){ .mmx = _mm_sr##aORl##_pi32(vec1.mmx, vec2.mmx) }; \
	} while (0)

// shared between MMX variations
#define VEC_MMX_MUL_8x8(sign) \
	VEC_MMX_OPERATION_8x8(mullo, sign)

#define VEC_MMX_MUL_16x4(sign) \
	do { \
		/* we have a real instruction for this */ \
		return (v##sign##int16x4){ .mmx = _mm_mullo_pi16(vec1.mmx, vec2.mmx) }; \
	} while (0)

#define VEC_MMX_MUL_32x2(sign) \
	do { \
		__m64 ac = _mm_mullo_pi16(vec1.mmx, vec2.mmx); \
		__m64 b  = _mm_srli_pi32(vec1.mmx, 16); \
		__m64 bc = _mm_mullo_pi16(b, vec2.mmx); \
		__m64 d  = _mm_srli_pi32(vec2.mmx, 16); \
		__m64 ad = _mm_mullo_pi16(vec1.mmx, d); \
		__m64 hi = _mm_add_pi32(bc, ad); \
		hi = _mm_slli_pi32(hi, 16); \
		return (v##sign##int32x2) { .mmx = _mm_add_pi32(hi, ac) }; /* return ac + hi; */ \
	} while (0)

#define VEC_MMX_DEFINE_OPERATIONS_SIGN(sign, bits, size) \
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_load_aligned(const sign##int##bits##_t in[size]) \
	{ \
		v##sign##int##bits##x##size vec; \
		memcpy(&vec.mmx, in, sizeof(vec.mmx)); \
		return vec; \
	} \
	\
	static void v##sign##int##bits##x##size##_mmx_store_aligned(v##sign##int##bits##x##size vec, sign##int##bits##_t out[size]) \
	{ \
		memcpy(out, &vec.mmx, sizeof(vec.mmx)); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_add(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .mmx = _mm_add_pi##bits(vec1.mmx, vec2.mmx) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_sub(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .mmx = _mm_sub_pi##bits(vec1.mmx, vec2.mmx) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_mul(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		VEC_MMX_MUL_##bits##x##size(sign); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_and(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .mmx = _mm_and_si64(vec1.mmx, vec2.mmx) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_or(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .mmx = _mm_or_si64(vec1.mmx, vec2.mmx) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_xor(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .mmx = _mm_xor_si64(vec1.mmx, vec2.mmx) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_lshift(v##sign##int##bits##x##size vec1, vuint##bits##x##size vec2) \
	{ \
		VEC_MMX_LSHIFT_##bits##x##size(sign); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_rshift(v##sign##int##bits##x##size vec1, vuint##bits##x##size vec2) \
	{ \
		VEC_MMX_RSHIFT_##bits##x##size(sign, a); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_mmx_lrshift(v##sign##int##bits##x##size vec1, vuint##bits##x##size vec2) \
	{ \
		VEC_MMX_RSHIFT_##bits##x##size(sign, l); \
	} \
	\
	static v##sign##int##bits##x##size##_impl v##sign##int##bits##x##size##_impl_mmx = { \
		.load_aligned  = v##sign##int##bits##x##size##_mmx_load_aligned, \
		.load          = v##sign##int##bits##x##size##_mmx_load_aligned, \
		.store_aligned = v##sign##int##bits##x##size##_mmx_store_aligned, \
		.store         = v##sign##int##bits##x##size##_mmx_store_aligned, \
		.add           = v##sign##int##bits##x##size##_mmx_add, \
		.sub           = v##sign##int##bits##x##size##_mmx_sub, \
		.mul           = v##sign##int##bits##x##size##_mmx_mul, \
		.and           = v##sign##int##bits##x##size##_mmx_and, \
		.or            = v##sign##int##bits##x##size##_mmx_or, \
		.xor           = v##sign##int##bits##x##size##_mmx_xor, \
		.lshift        = v##sign##int##bits##x##size##_mmx_lshift, \
		.rshift        = v##sign##int##bits##x##size##_mmx_rshift, \
		.lrshift       = v##sign##int##bits##x##size##_mmx_lrshift, \
	};

#define VEC_MMX_DEFINE_OPERATIONS(bits, size) \
	VEC_MMX_DEFINE_OPERATIONS_SIGN( , bits, size) \
	VEC_MMX_DEFINE_OPERATIONS_SIGN(u, bits, size)

VEC_MMX_DEFINE_OPERATIONS(8, 8)
VEC_MMX_DEFINE_OPERATIONS(16, 4)
VEC_MMX_DEFINE_OPERATIONS(32, 2)

#undef VEC_MMX_DEFINE_OPERATIONS
#undef VEC_MMX_DEFINE_OPERATIONS_SIGN
#undef VEC_MMX_MUL_8x8
#undef VEC_MMX_MUL_16x4
#undef VEC_MMX_MUL_32x2
#undef VEC_MMX_OPERATION_8x8
#undef VEC_MMX_LSHIFT_8x8
#undef VEC_MMX_LSHIFT_16x4
#undef VEC_MMX_LSHIFT_32x2
#undef VEC_MMX_RSHIFT_8x8
#undef VEC_MMX_RSHIFT_16x4
#undef VEC_MMX_RSHIFT_32x2

#endif /* VEC_IMPL_X86_MMX_H_ */
