/**
 * vec - a tiny SIMD vector library in C99
 * 
 * Copyright (c) 2024-2025 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

/* Use this file to generate include/vec/impl/double.h !!
 *
 * `gcc -o gendouble gendouble.c` */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#define ARRAY_SIZE(x) (sizeof(x)/sizeof((x)[0]))

/* XXX: would it be faster to unroll literally everything instead of defining everything,
 * and then unpacking it all? */
static const char *header =
	"/**\n"
	" * vec - a tiny SIMD vector library in C99\n"
	" * \n"
	" * Copyright (c) 2024-2025 Paper\n"
	" * \n"
	" * Permission is hereby granted, free of charge, to any person obtaining a copy\n"
	" * of this software and associated documentation files (the \"Software\"), to deal\n"
	" * in the Software without restriction, including without limitation the rights\n"
	" * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell\n"
	" * copies of the Software, and to permit persons to whom the Software is\n"
	" * furnished to do so, subject to the following conditions:\n"
	" * \n"
	" * The above copyright notice and this permission notice shall be included in all\n"
	" * copies or substantial portions of the Software.\n"
	" * \n"
	" * THE SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR\n"
	" * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,\n"
	" * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE\n"
	" * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER\n"
	" * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,\n"
	" * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE\n"
	" * SOFTWARE.\n"
	"**/\n"
	"\n"
	"/* This file is automatically generated! Do not edit it directly!\n"
	" * Edit the code that generates it in utils/gengeneric.c  --paper */\n"
	"\n"
	"#ifndef VEC_IMPL_DOUBLE_H_\n"
	"#define VEC_IMPL_DOUBLE_H_\n"
	"\n"
	"#define VEC_DOUBLE_SPLAT(sign, bits, size, halfsize) \\\n"
	"	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_splat(vec_##sign##int##bits x) \\\n"
	"	{ \\\n"
	"		v##sign##int##bits##x##size vec; \\\n"
	"	\\\n"
	"		vec.generic[0] = v##sign##int##bits##x##halfsize##_splat(x); \\\n"
	"		vec.generic[1] = v##sign##int##bits##x##halfsize##_splat(x); \\\n"
	"	\\\n"
	"		return vec; \\\n"
	"	}\n"
	"\n"
	"#define VEC_DOUBLE_LOAD_EX(name, sign, bits, size, halfsize) \\\n"
	"	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_##name(const vec_##sign##int##bits x[size]) \\\n"
	"	{ \\\n"
	"		v##sign##int##bits##x##size vec; \\\n"
	"	\\\n"
	"		vec.generic[0] = v##sign##int##bits##x##halfsize##_##name(x); \\\n"
	"		vec.generic[1] = v##sign##int##bits##x##halfsize##_##name(x + halfsize); \\\n"
	"	\\\n"
	"		return vec; \\\n"
	"	}\n"
	"\n"
	"#define VEC_DOUBLE_LOAD(sign, bits, size, halfsize) VEC_DOUBLE_LOAD_EX(load, sign, bits, size, halfsize)\n"
	"#define VEC_DOUBLE_LOAD_ALIGNED(sign, bits, size, halfsize) VEC_DOUBLE_LOAD_EX(load_aligned, sign, bits, size, halfsize)\n"
	"\n"
	"#define VEC_DOUBLE_STORE_EX(name, sign, bits, size, halfsize) \\\n"
	"	VEC_FUNC_IMPL void v##sign##int##bits##x##size##_##name(v##sign##int##bits##x##size vec, vec_##sign##int##bits x[size]) \\\n"
	"	{ \\\n"
	"		v##sign##int##bits##x##halfsize##_##name(vec.generic[0], x); \\\n"
	"		v##sign##int##bits##x##halfsize##_##name(vec.generic[1], x + halfsize); \\\n"
	"	}\n"
	"\n"
	"#define VEC_DOUBLE_STORE(sign, bits, size, halfsize) VEC_DOUBLE_STORE_EX(store, sign, bits, size, halfsize)\n"
	"#define VEC_DOUBLE_STORE_ALIGNED(sign, bits, size, halfsize) VEC_DOUBLE_STORE_EX(store_aligned, sign, bits, size, halfsize)\n"
	"\n"
	"#define VEC_DOUBLE_OP(name, sign, bits, size, halfsize, secondsign) \\\n"
	"	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_##name(v##sign##int##bits##x##size vec1, v##secondsign##int##bits##x##size vec2) \\\n"
	"	{ \\\n"
	"		vec1.generic[0] = v##sign##int##bits##x##halfsize##_##name(vec1.generic[0], vec2.generic[0]); \\\n"
	"		vec1.generic[1] = v##sign##int##bits##x##halfsize##_##name(vec1.generic[1], vec2.generic[1]); \\\n"
	"	\\\n"
	"		return vec1; \\\n"
	"	}\n"
	"\n"
	"#define VEC_DOUBLE_ADD(sign, bits, size, halfsize) VEC_DOUBLE_OP(add, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_SUB(sign, bits, size, halfsize) VEC_DOUBLE_OP(sub, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_MUL(sign, bits, size, halfsize) VEC_DOUBLE_OP(mul, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_DIV(sign, bits, size, halfsize) VEC_DOUBLE_OP(div, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_MOD(sign, bits, size, halfsize) VEC_DOUBLE_OP(mod, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_AVG(sign, bits, size, halfsize) VEC_DOUBLE_OP(avg, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_LSHIFT(sign, bits, size, halfsize) VEC_DOUBLE_OP(lshift, sign, bits, size, halfsize, u)\n"
	"#define VEC_DOUBLE_RSHIFT(sign, bits, size, halfsize) VEC_DOUBLE_OP(rshift, sign, bits, size, halfsize, u)\n"
	"#define VEC_DOUBLE_LRSHIFT(sign, bits, size, halfsize) VEC_DOUBLE_OP(lrshift, sign, bits, size, halfsize, u)\n"
	"#define VEC_DOUBLE_AND(sign, bits, size, halfsize) VEC_DOUBLE_OP(and, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_OR(sign, bits, size, halfsize) VEC_DOUBLE_OP(or, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_XOR(sign, bits, size, halfsize) VEC_DOUBLE_OP(xor, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_MIN(sign, bits, size, halfsize) VEC_DOUBLE_OP(min, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_MAX(sign, bits, size, halfsize) VEC_DOUBLE_OP(max, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_CMPLT(sign, bits, size, halfsize) VEC_DOUBLE_OP(cmplt, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_CMPLE(sign, bits, size, halfsize) VEC_DOUBLE_OP(cmple, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_CMPEQ(sign, bits, size, halfsize) VEC_DOUBLE_OP(cmpeq, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_CMPGE(sign, bits, size, halfsize) VEC_DOUBLE_OP(cmpge, sign, bits, size, halfsize, sign)\n"
	"#define VEC_DOUBLE_CMPGT(sign, bits, size, halfsize) VEC_DOUBLE_OP(cmpgt, sign, bits, size, halfsize, sign)\n"
	"\n"
	"#define VEC_DOUBLE_NOT(sign, bits, size, halfsize) \\\n"
	"	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_not(v##sign##int##bits##x##size vec) \\\n"
	"	{ \\\n"
	"		vec.generic[0] = v##sign##int##bits##x##halfsize##_not(vec.generic[0]); \\\n"
	"		vec.generic[1] = v##sign##int##bits##x##halfsize##_not(vec.generic[1]); \\\n"
	"	\\\n"
	"		return vec; \\\n"
	"	}\n"
	"\n"
	"#endif /* VEC_IMPL_DOUBLE_H_ */ \n"
	"\n"
	"/* ------------------------------------------------------------------------ */\n"
	"/* PREPROCESSOR HELL INCOMING */\n"
	"";

static const char *footer = 
	"" /* nothing */;

/* ------------------------------------------------------------------------ */

static void print_generic_dbl_op(const char *op, int is_signed, int bits, int size)
{
	printf(
		"#if !defined(V%sINT%dx%d_%s_DEFINED) && defined(V%sINT%dx%d_%s_DEFINED)\n"
		"VEC_DOUBLE_%s(%s, %d, %d, %d)\n"
		"# define V%sINT%dx%d_%s_DEFINED\n"
		"#endif\n\n",
	(is_signed ? "" : "U"), bits, size, op, (is_signed ? "" : "U"), bits, size / 2, op,
	op, (is_signed ? "/* nothing */" : "u"), bits, size, size / 2,
	(is_signed ? "" : "U"), bits, size, op);
}

typedef void (*print_op_spec)(const char *op, int is_signed, int bits, int size);

static inline void print_ops(int is_signed, int bits, int size, print_op_spec print_op)
{
	/* all supported operations here */
	static const char *ops[] = {
		"SPLAT",
		"LOAD_ALIGNED",
		"LOAD",
		"STORE_ALIGNED",
		"STORE",
		"ADD",
		"SUB",
		"MUL",
		"DIV",
		"MOD",
		"AVG",
		"AND",
		"OR",
		"XOR",
		"NOT",
		"CMPLT",
		"CMPEQ",
		"CMPGT",
		"CMPLE", /* these two must be after CMPLT and CMPGT respectfully, */
		"CMPGE", /* because their definitions call those functions */
		"MIN",
		"MAX",
		"RSHIFT",
		"LRSHIFT",
		"LSHIFT",
		NULL,
	};
	int i;

	printf("\n\n/* v%sint%dx%d */\n\n", (is_signed ? "u" : ""), bits, size);

	for (i = 0; ops[i]; i++)
		print_op(ops[i], is_signed, bits, size);
}

int main(void)
{
	static struct {
		int bits, size;
		print_op_spec print_op;
	} defs[] = {
		/* -- 8-bit */
		{8, 4, print_generic_dbl_op},
		{8, 8, print_generic_dbl_op},
		{8, 16, print_generic_dbl_op},
		{8, 32, print_generic_dbl_op},
		{8, 64, print_generic_dbl_op},

		/* -- 16-bit */
		{16, 4, print_generic_dbl_op},
		{16, 8, print_generic_dbl_op},
		{16, 16, print_generic_dbl_op},
		{16, 32, print_generic_dbl_op},

		/* -- 32-bit */
		{32, 4, print_generic_dbl_op},
		{32, 8, print_generic_dbl_op},
		{32, 16, print_generic_dbl_op},

		/* -- 64-bit */
		{64, 4, print_generic_dbl_op},
		{64, 8, print_generic_dbl_op},
	};
	int i;

	puts(header);

	for (i = 0; i < ARRAY_SIZE(defs); i++) {
		print_ops(1, defs[i].bits, defs[i].size, defs[i].print_op);
		print_ops(0, defs[i].bits, defs[i].size, defs[i].print_op);
	}

	puts(footer);
}
