/**
 * vec - a tiny SIMD vector library in C99
 * 
 * Copyright (c) 2024-2025 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

#ifndef VEC_IMPL_X86_AVX2_H_
#define VEC_IMPL_X86_AVX2_H_

#define VEC_AVX2_OP(NAME, SIGN, BITS, SIZE, INTLSIGN, OP) \
	VEC_FUNC_IMPL v##SIGN##int##BITS##x##SIZE v##SIGN##int##BITS##x##SIZE##_##NAME(v##SIGN##int##BITS##x##SIZE vec1, v##SIGN##int##BITS##x##SIZE vec2) \
	{ \
		vec1.avx2 = _mm256_##OP##_ep##INTLSIGN##BITS(vec1.avx2, vec2.avx2); \
	\
		return vec1; \
	}

#ifndef VINT8x32_MIN_DEFINED
VEC_AVX2_OP(min, /* nothing */, 8, 32, i, min)
# define VINT8x32_MIN_DEFINED
#endif

#ifndef VINT8x32_MAX_DEFINED
VEC_AVX2_OP(max, /* nothing */, 8, 32, i, max)
# define VINT8x32_MAX_DEFINED
#endif

#ifndef VUINT8x32_MIN_DEFINED
VEC_AVX2_OP(min, u, 8, 32, u, min)
# define VUINT8x32_MIN_DEFINED
#endif

#ifndef VUINT8x32_MAX_DEFINED
VEC_AVX2_OP(max, u, 8, 32, u, max)
# define VUINT8x32_MAX_DEFINED
#endif

#ifndef VINT16x16_MIN_DEFINED
VEC_AVX2_OP(min, /* nothing */, 16, 16, i, min)
# define VINT16x16_MIN_DEFINED
#endif

#ifndef VINT16x16_MAX_DEFINED
VEC_AVX2_OP(max, /* nothing */, 16, 16, i, max)
# define VINT16x16_MAX_DEFINED
#endif

#ifndef VUINT16x16_MIN_DEFINED
VEC_AVX2_OP(min, u, 16, 16, u, min)
# define VUINT16x16_MIN_DEFINED
#endif

#ifndef VUINT16x16_MAX_DEFINED
VEC_AVX2_OP(max, u, 16, 16, u, max)
# define VUINT16x16_MAX_DEFINED
#endif

#ifndef VINT32x8_MIN_DEFINED
VEC_AVX2_OP(min, /* nothing */, 32, 8, i, min)
# define VINT32x8_MIN_DEFINED
#endif

#ifndef VINT32x8_MAX_DEFINED
VEC_AVX2_OP(max, /* nothing */, 32, 8, i, max)
# define VINT32x8_MAX_DEFINED
#endif

#ifndef VUINT32x8_MIN_DEFINED
VEC_AVX2_OP(min, u, 32, 8, u, min)
# define VUINT32x8_MIN_DEFINED
#endif

#ifndef VUINT32x8_MAX_DEFINED
VEC_AVX2_OP(max, u, 32, 8, u, max)
# define VUINT32x8_MAX_DEFINED
#endif

/* ------------------------------------------------------------------------ */
/* multiplication */

#ifndef VINT16x16_MUL_DEFINED
VEC_AVX2_OP(mul, /* nothing */, 16, 16, i, mullo)
# define VINT16x16_MUL_DEFINED
#endif

#ifndef VUINT16x16_MUL_DEFINED
VEC_AVX2_OP(mul, u, 16, 16, i, mullo)
# define VUINT16x16_MUL_DEFINED
#endif

#ifndef VINT32x8_MUL_DEFINED
VEC_AVX2_OP(mul, /* nothing */, 32, 8, i, mullo)
# define VINT32x8_MUL_DEFINED
#endif

#ifndef VUINT32x8_MUL_DEFINED
VEC_AVX2_OP(mul, u, 32, 8, i, mullo)
# define VUINT32x8_MUL_DEFINED
#endif

#define VEC_AVX2_MUL_64x4(sign) \
	VEC_FUNC_IMPL v##sign##int64x4 v##sign##int64x4_mul(v##sign##int64x4 vec1, v##sign##int64x4 vec2) \
	{ \
		__m256i ac = _mm256_mul_epu32(vec1.avx2, vec2.avx2); \
		__m256i b  = _mm256_srli_epi64(vec1.avx2, 32); \
		__m256i bc = _mm256_mul_epu32(b, vec2.avx2); \
		__m256i d  = _mm256_srli_epi64(vec2.avx2, 32); \
		__m256i ad = _mm256_mul_epu32(vec1.avx2, d); \
		__m256i hi = _mm256_add_epi64(bc, ad); \
		hi = _mm256_slli_epi64(hi, 32); \
	\
		vec1.avx2 = _mm256_add_epi64(hi, ac); \
		return vec1; \
	}

#ifndef VINT64x4_MUL_DEFINED
VEC_AVX2_MUL_64x4(/* nothing */)
# define VINT64x4_MUL_DEFINED
#endif

#ifndef VUINT64x4_MUL_DEFINED
VEC_AVX2_MUL_64x4(u)
# define VUINT64x4_MUL_DEFINED
#endif

/* -------------------------------------------------------------------- */
/* avg */

#ifndef VUINT8x32_AVG_DEFINED
VEC_AVX2_OP(avg, u, 8, 32, u, avg)
# define VUINT8x32_AVG_DEFINED
#endif

#ifndef VUINT16x16_AVG_DEFINED
VEC_AVX2_OP(avg, u, 16, 16, u, avg)
# define VUINT16x16_AVG_DEFINED
#endif

/* -------------------------------------------------------------------- */
/* generic ops */

#define VEC_AVX2_SPLAT(sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_splat(vec_##sign##int##bits x) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.avx2 = _mm256_set1_epi##bits(x); \
		return vec; \
	}

#define VEC_AVX2_LOAD_ALIGNED(sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_load_aligned(const vec_##sign##int##bits in[size]) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.avx2 = _mm256_load_si256((const __m256i *)in); \
		return vec; \
	}

#define VEC_AVX2_LOAD(sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_load(const vec_##sign##int##bits in[size]) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.avx2 = _mm256_loadu_si256((const __m256i *)in); \
		return vec; \
	}

#define VEC_AVX2_STORE_ALIGNED(sign, bits, size) \
	VEC_FUNC_IMPL void v##sign##int##bits##x##size##_store_aligned(v##sign##int##bits##x##size vec, vec_##sign##int##bits out[size]) \
	{ \
		_mm256_store_si256((__m256i *)out, vec.avx2); \
	}

#define VEC_AVX2_STORE(sign, bits, size) \
	VEC_FUNC_IMPL void v##sign##int##bits##x##size##_store(v##sign##int##bits##x##size vec, vec_##sign##int##bits out[size]) \
	{ \
		_mm256_storeu_si256((__m256i *)out, vec.avx2); \
	}

#define VEC_AVX2_BITWISE(op, sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_##op(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		vec1.avx2 = _mm256_##op##_si256(vec1.avx2, vec2.avx2); \
		return vec1; \
	}

/* ------------------------------------------------------------------------ */
/* 8x32 */

#ifndef VINT8x32_SPLAT_DEFINED
VEC_AVX2_SPLAT(/* nothing */, 8, 32)
# define VINT8x32_SPLAT_DEFINED
#endif

#ifndef VINT8x32_LOAD_ALIGNED_DEFINED
VEC_AVX2_LOAD_ALIGNED(/* nothing */, 8, 32)
# define VINT8x32_LOAD_ALIGNED_DEFINED
#endif

#ifndef VINT8x32_LOAD_DEFINED
VEC_AVX2_LOAD(/* nothing */, 8, 32)
# define VINT8x32_LOAD_DEFINED
#endif

#ifndef VINT8x32_STORE_ALIGNED_DEFINED
VEC_AVX2_STORE_ALIGNED(/* nothing */, 8, 32)
# define VINT8x32_STORE_ALIGNED_DEFINED
#endif

#ifndef VINT8x32_STORE_DEFINED
VEC_AVX2_STORE(/* nothing */, 8, 32)
# define VINT8x32_STORE_DEFINED
#endif

#ifndef VINT8x32_ADD_DEFINED
VEC_AVX2_OP(add, /* nothing */, 8, 32, i, add)
# define VINT8x32_ADD_DEFINED
#endif

#ifndef VINT8x32_SUB_DEFINED
VEC_AVX2_OP(sub, /* nothing */, 8, 32, i, sub)
# define VINT8x32_SUB_DEFINED
#endif

#ifndef VINT8x32_AND_DEFINED
VEC_AVX2_BITWISE(and, /* nothing */, 8, 32)
# define VINT8x32_AND_DEFINED
#endif

#ifndef VINT8x32_OR_DEFINED
VEC_AVX2_BITWISE(or, /* nothing */, 8, 32)
# define VINT8x32_OR_DEFINED
#endif

#ifndef VINT8x32_XOR_DEFINED
VEC_AVX2_BITWISE(xor, /* nothing */, 8, 32)
# define VINT8x32_XOR_DEFINED
#endif

/* u8x32 */

#ifndef VUINT8x32_SPLAT_DEFINED
VEC_AVX2_SPLAT(u, 8, 32)
# define VUINT8x32_SPLAT_DEFINED
#endif

#ifndef VUINT8x32_LOAD_ALIGNED_DEFINED
VEC_AVX2_LOAD_ALIGNED(u, 8, 32)
# define VUINT8x32_LOAD_ALIGNED_DEFINED
#endif

#ifndef VUINT8x32_LOAD_DEFINED
VEC_AVX2_LOAD(u, 8, 32)
# define VUINT8x32_LOAD_DEFINED
#endif

#ifndef VUINT8x32_STORE_ALIGNED_DEFINED
VEC_AVX2_STORE_ALIGNED(u, 8, 32)
# define VUINT8x32_STORE_ALIGNED_DEFINED
#endif

#ifndef VUINT8x32_STORE_DEFINED
VEC_AVX2_STORE(u, 8, 32)
# define VUINT8x32_STORE_DEFINED
#endif

#ifndef VUINT8x32_ADD_DEFINED
VEC_AVX2_OP(add, u, 8, 32, i, add)
# define VUINT8x32_ADD_DEFINED
#endif

#ifndef VUINT8x32_SUB_DEFINED
VEC_AVX2_OP(sub, u, 8, 32, i, sub)
# define VUINT8x32_SUB_DEFINED
#endif

#ifndef VUINT8x32_AND_DEFINED
VEC_AVX2_BITWISE(and, u, 8, 32)
# define VUINT8x32_AND_DEFINED
#endif

#ifndef VUINT8x32_OR_DEFINED
VEC_AVX2_BITWISE(or, u, 8, 32)
# define VUINT8x32_OR_DEFINED
#endif

#ifndef VUINT8x32_XOR_DEFINED
VEC_AVX2_BITWISE(xor, u, 8, 32)
# define VUINT8x32_XOR_DEFINED
#endif

/* ------------------------------------------------------------------------ */
/* 16x16 */

#ifndef VINT16x16_SPLAT_DEFINED
VEC_AVX2_SPLAT(/* nothing */, 16, 16)
# define VINT16x16_SPLAT_DEFINED
#endif

#ifndef VINT16x16_LOAD_ALIGNED_DEFINED
VEC_AVX2_LOAD_ALIGNED(/* nothing */, 16, 16)
# define VINT16x16_LOAD_ALIGNED_DEFINED
#endif

#ifndef VINT16x16_LOAD_DEFINED
VEC_AVX2_LOAD(/* nothing */, 16, 16)
# define VINT16x16_LOAD_DEFINED
#endif

#ifndef VINT16x16_STORE_ALIGNED_DEFINED
VEC_AVX2_STORE_ALIGNED(/* nothing */, 16, 16)
# define VINT16x16_STORE_ALIGNED_DEFINED
#endif

#ifndef VINT16x16_STORE_DEFINED
VEC_AVX2_STORE(/* nothing */, 16, 16)
# define VINT16x16_STORE_DEFINED
#endif

#ifndef VINT16x16_ADD_DEFINED
VEC_AVX2_OP(add, /* nothing */, 16, 16, i, add)
# define VINT16x16_ADD_DEFINED
#endif

#ifndef VINT16x16_SUB_DEFINED
VEC_AVX2_OP(sub, /* nothing */, 16, 16, i, sub)
# define VINT16x16_SUB_DEFINED
#endif

#ifndef VINT16x16_AND_DEFINED
VEC_AVX2_BITWISE(and, /* nothing */, 16, 16)
# define VINT16x16_AND_DEFINED
#endif

#ifndef VINT16x16_OR_DEFINED
VEC_AVX2_BITWISE(or, /* nothing */, 16, 16)
# define VINT16x16_OR_DEFINED
#endif

#ifndef VINT16x16_XOR_DEFINED
VEC_AVX2_BITWISE(xor, /* nothing */, 16, 16)
# define VINT16x16_XOR_DEFINED
#endif

/* u16x16 */

#ifndef VUINT16x16_SPLAT_DEFINED
VEC_AVX2_SPLAT(u, 16, 16)
# define VUINT16x16_SPLAT_DEFINED
#endif

#ifndef VUINT16x16_LOAD_ALIGNED_DEFINED
VEC_AVX2_LOAD_ALIGNED(u, 16, 16)
# define VUINT16x16_LOAD_ALIGNED_DEFINED
#endif

#ifndef VUINT16x16_LOAD_DEFINED
VEC_AVX2_LOAD(u, 16, 16)
# define VUINT16x16_LOAD_DEFINED
#endif

#ifndef VUINT16x16_STORE_ALIGNED_DEFINED
VEC_AVX2_STORE_ALIGNED(u, 16, 16)
# define VUINT16x16_STORE_ALIGNED_DEFINED
#endif

#ifndef VUINT16x16_STORE_DEFINED
VEC_AVX2_STORE(u, 16, 16)
# define VUINT16x16_STORE_DEFINED
#endif

#ifndef VUINT16x16_ADD_DEFINED
VEC_AVX2_OP(add, u, 16, 16, i, add)
# define VUINT16x16_ADD_DEFINED
#endif

#ifndef VUINT16x16_SUB_DEFINED
VEC_AVX2_OP(sub, u, 16, 16, i, sub)
# define VUINT16x16_SUB_DEFINED
#endif

#ifndef VUINT16x16_AND_DEFINED
VEC_AVX2_BITWISE(and, u, 16, 16)
# define VUINT16x16_AND_DEFINED
#endif

#ifndef VUINT16x16_OR_DEFINED
VEC_AVX2_BITWISE(or, u, 16, 16)
# define VUINT16x16_OR_DEFINED
#endif

#ifndef VUINT16x16_XOR_DEFINED
VEC_AVX2_BITWISE(xor, u, 16, 16)
# define VUINT16x16_XOR_DEFINED
#endif

/* ------------------------------------------------------------------------ */
/* 32x8 */

#ifndef VINT32x8_SPLAT_DEFINED
VEC_AVX2_SPLAT(/* nothing */, 32, 8)
# define VINT32x8_SPLAT_DEFINED
#endif

#ifndef VINT32x8_LOAD_ALIGNED_DEFINED
VEC_AVX2_LOAD_ALIGNED(/* nothing */, 32, 8)
# define VINT32x8_LOAD_ALIGNED_DEFINED
#endif

#ifndef VINT32x8_LOAD_DEFINED
VEC_AVX2_LOAD(/* nothing */, 32, 8)
# define VINT32x8_LOAD_DEFINED
#endif

#ifndef VINT32x8_STORE_ALIGNED_DEFINED
VEC_AVX2_STORE_ALIGNED(/* nothing */, 32, 8)
# define VINT32x8_STORE_ALIGNED_DEFINED
#endif

#ifndef VINT32x8_STORE_DEFINED
VEC_AVX2_STORE(/* nothing */, 32, 8)
# define VINT32x8_STORE_DEFINED
#endif

#ifndef VINT32x8_ADD_DEFINED
VEC_AVX2_OP(add, /* nothing */, 32, 8, i, add)
# define VINT32x8_ADD_DEFINED
#endif

#ifndef VINT32x8_SUB_DEFINED
VEC_AVX2_OP(sub, /* nothing */, 32, 8, i, sub)
# define VINT32x8_SUB_DEFINED
#endif

#ifndef VINT32x8_AND_DEFINED
VEC_AVX2_BITWISE(and, /* nothing */, 32, 8)
# define VINT32x8_AND_DEFINED
#endif

#ifndef VINT32x8_OR_DEFINED
VEC_AVX2_BITWISE(or, /* nothing */, 32, 8)
# define VINT32x8_OR_DEFINED
#endif

#ifndef VINT32x8_XOR_DEFINED
VEC_AVX2_BITWISE(xor, /* nothing */, 32, 8)
# define VINT32x8_XOR_DEFINED
#endif

/* u32x8 */

#ifndef VUINT32x8_SPLAT_DEFINED
VEC_AVX2_SPLAT(u, 32, 8)
# define VUINT32x8_SPLAT_DEFINED
#endif

#ifndef VUINT32x8_LOAD_ALIGNED_DEFINED
VEC_AVX2_LOAD_ALIGNED(u, 32, 8)
# define VUINT32x8_LOAD_ALIGNED_DEFINED
#endif

#ifndef VUINT32x8_LOAD_DEFINED
VEC_AVX2_LOAD(u, 32, 8)
# define VUINT32x8_LOAD_DEFINED
#endif

#ifndef VUINT32x8_STORE_ALIGNED_DEFINED
VEC_AVX2_STORE_ALIGNED(u, 32, 8)
# define VUINT32x8_STORE_ALIGNED_DEFINED
#endif

#ifndef VUINT32x8_STORE_DEFINED
VEC_AVX2_STORE(u, 32, 8)
# define VUINT32x8_STORE_DEFINED
#endif

#ifndef VUINT32x8_ADD_DEFINED
VEC_AVX2_OP(add, u, 32, 8, i, add)
# define VUINT32x8_ADD_DEFINED
#endif

#ifndef VUINT32x8_SUB_DEFINED
VEC_AVX2_OP(sub, u, 32, 8, i, sub)
# define VUINT32x8_SUB_DEFINED
#endif

#ifndef VUINT32x8_AND_DEFINED
VEC_AVX2_BITWISE(and, u, 32, 8)
# define VUINT32x8_AND_DEFINED
#endif

#ifndef VUINT32x8_OR_DEFINED
VEC_AVX2_BITWISE(or, u, 32, 8)
# define VUINT32x8_OR_DEFINED
#endif

#ifndef VUINT32x8_XOR_DEFINED
VEC_AVX2_BITWISE(xor, u, 32, 8)
# define VUINT32x8_XOR_DEFINED
#endif

#endif /* VEC_IMPL_X86_AVX2_H_ */
