/**
 * vec - a tiny SIMD vector library in C99
 * 
 * Copyright (c) 2024 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

#ifndef VEC_IMPL_X86_AVX512F_H_
#define VEC_IMPL_X86_AVX512F_H_

#define VEC_AVX512F_OPERATION_8x64(op, sign) \
	do { \
		/* unpack and add */ \
		__m512i dst_1 = _mm512_##op##_epi32(vec1.avx512f, vec2.avx512f); \
		__m512i dst_2 = _mm512_##op##_epi32(_mm512_srli_epi32(vec1.avx512f, 8), _mm512_srli_epi32(vec2.avx512f, 8)); \
		__m512i dst_3 = _mm512_##op##_epi32(_mm512_srli_epi32(vec1.avx512f, 16), _mm512_srli_epi32(vec2.avx512f, 16)); \
		__m512i dst_4 = _mm512_##op##_epi32(_mm512_srli_epi32(vec1.avx512f, 24), _mm512_srli_epi32(vec2.avx512f, 24)); \
	\
		/* repack */ \
		return (v##sign##int8x64){ .avx512f = _mm512_or_si512( \
			_mm512_or_si512( \
				_mm512_slli_epi32(dst_4, 8), \
				_mm512_srli_epi32(_mm512_slli_epi32(dst_3, 8), 8) \
			), \
			_mm512_or_si512( \
				_mm512_slli_epi32(_mm512_slli_epi32(dst_2, 8), 16), \
				_mm512_srli_epi32(_mm512_slli_epi32(dst_1, 8), 24) \
			) \
		)}; \
	} while (0)

#define VEC_AVX512F_OPERATION_16x32(op, sign) \
	do { \
		/* unpack and add */ \
		__m512i dst_even = _mm512_##op##_epi32(vec1.avx512f, vec2.avx512f); \
		__m512i dst_odd = _mm512_##op##_epi32(_mm512_srli_epi32(vec1.avx512f, 16), _mm512_srli_epi32(vec2.avx512f, 16)); \
	\
		/* repack */ \
		return (v##sign##int16x32){ .avx512f = _mm512_or_si512( \
			_mm512_slli_epi32(dst_odd, 16), \
			_mm512_srli_epi32(_mm512_slli_epi32(dst_even, 16), 16) \
		)}; \
	} while (0)

#define VEC_AVX512F_ADD_8x64(sign) \
	VEC_AVX512F_OPERATION_8x64(add, sign)

#define VEC_AVX512F_ADD_16x32(sign) \
	VEC_AVX512F_OPERATION_16x32(add, sign)

#define VEC_AVX512F_ADD_32x16(sign) \
	do { \
		return (v##sign##int32x16) { .avx512f = _mm512_add_epi32(vec1.avx512f, vec2.avx512f) }; \
	} while (0)

#define VEC_AVX512F_ADD_64x8(sign) \
	do { \
		return (v##sign##int64x8) { .avx512f = _mm512_add_epi64(vec1.avx512f, vec2.avx512f) }; \
	} while (0)

#define VEC_AVX512F_SUB_8x64(sign) \
	VEC_AVX512F_OPERATION_8x64(sub, sign)

#define VEC_AVX512F_SUB_16x32(sign) \
	VEC_AVX512F_OPERATION_16x32(sub, sign)

#define VEC_AVX512F_SUB_32x16(sign) \
	do { \
		return (v##sign##int32x16) { .avx512f = _mm512_sub_epi32(vec1.avx512f, vec2.avx512f) }; \
	} while (0)

#define VEC_AVX512F_SUB_64x8(sign) \
	do { \
		return (v##sign##int64x8) { .avx512f = _mm512_sub_epi64(vec1.avx512f, vec2.avx512f) }; \
	} while (0)

#define VEC_AVX512F_MUL_8x64(sign) \
	VEC_AVX512F_OPERATION_8x64(mullo, sign)

#define VEC_AVX512F_MUL_16x32(sign) \
	VEC_AVX512F_OPERATION_16x32(mullo, sign)

#define VEC_AVX512F_MUL_32x16(sign) \
	do { \
		return (v##sign##int32x16) { .avx512f = _mm512_mullo_epi32(vec1.avx512f, vec2.avx512f) }; \
	} while (0)

#define VEC_AVX512F_MUL_64x8(sign) \
	do { \
		__m512i ac = _mm512_mul_epu32(vec1.avx512f, vec2.avx512f); \
		__m512i b  = _mm512_srli_epi64(vec1.avx512f, 32); \
		__m512i bc = _mm512_mul_epu32(b, vec2.avx512f); \
		__m512i d  = _mm512_srli_epi64(vec2.avx512f, 32); \
		__m512i ad = _mm512_mul_epu32(vec1.avx512f, d); \
		__m512i hi = _mm512_add_epi64(bc, ad); \
		hi = _mm512_slli_epi64(hi, 32); \
		return (v##sign##int64x8) { .avx512f = _mm512_add_epi64(hi, ac) }; \
	} while (0)

#define VEC_AVX512F_LSHIFT_8x64(sign) \
	VEC_AVX512F_OPERATION_8x64(sllv, sign)

#define VEC_AVX512F_LSHIFT_16x32(sign) \
	VEC_AVX512F_OPERATION_16x32(sllv, sign)

#define VEC_AVX512F_LSHIFT_32x16(sign) \
	do { \
		return (v##sign##int32x16){ .avx512f = _mm512_sllv_epi32(vec1.avx512f, vec2.avx512f) }; \
	} while (0)

#define VEC_AVX512F_LSHIFT_64x8(sign) \
	do { \
		return (v##sign##int64x8){ .avx512f = _mm512_sllv_epi64(vec1.avx512f, vec2.avx512f) }; \
	} while (0)

#define VEC_AVX512F_RSHIFT_8x64(sign, aORl) \
	VEC_AVX512F_OPERATION_8x64(sr##aORl##v, sign)

#define VEC_AVX512F_RSHIFT_16x32(sign, aORl) \
	VEC_AVX512F_OPERATION_16x32(sr##aORl##v, sign)

#define VEC_AVX512F_RSHIFT_32x16(sign, aORl) \
	do { \
		return (v##sign##int32x16){ .avx512f = _mm512_sr##aORl##v_epi32(vec1.avx512f, vec2.avx512f) }; \
	} while (0)

#define VEC_AVX512F_RSHIFT_64x8(sign, aORl) \
	do { \
		return (v##sign##int64x8){ .avx512f = _mm512_sr##aORl##v_epi64(vec1.avx512f, vec2.avx512f) }; \
	} while (0)

#define VEC_AVX512F_DEFINE_OPERATIONS_SIGN(sign, bits, size) \
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_load_aligned(const sign##int##bits##_t in[size]) \
	{ \
		return (v##sign##int##bits##x##size) { .avx512f = _mm512_load_si512((const __m512i *)in) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_load(const sign##int##bits##_t in[size]) \
	{ \
		return (v##sign##int##bits##x##size) { .avx512f = _mm512_loadu_si512((const __m512i *)in) }; \
	} \
	\
	static void v##sign##int##bits##x##size##_avx512f_store_aligned(v##sign##int##bits##x##size vec, sign##int##bits##_t out[size]) \
	{ \
		_mm512_store_si512((__m512i *)out, vec.avx512f); \
	} \
	\
	static void v##sign##int##bits##x##size##_avx512f_store(v##sign##int##bits##x##size vec, sign##int##bits##_t out[size]) \
	{ \
		_mm512_storeu_si512((__m512i *)out, vec.avx512f); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_add(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		VEC_AVX512F_ADD_##bits##x##size(sign); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_sub(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		VEC_AVX512F_SUB_##bits##x##size(sign); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_mul(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		VEC_AVX512F_MUL_##bits##x##size(sign); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_and(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .avx512f = _mm512_and_si512(vec1.avx512f, vec2.avx512f) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_or(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .avx512f = _mm512_or_si512(vec1.avx512f, vec2.avx512f) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_xor(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		return (v##sign##int##bits##x##size) { .avx512f = _mm512_xor_si512(vec1.avx512f, vec2.avx512f) }; \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_lshift(v##sign##int##bits##x##size vec1, vuint##bits##x##size vec2) \
	{ \
		VEC_AVX512F_LSHIFT_##bits##x##size(sign); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_rshift(v##sign##int##bits##x##size vec1, vuint##bits##x##size vec2) \
	{ \
		VEC_AVX512F_RSHIFT_##bits##x##size(sign, a); \
	} \
	\
	static v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_lrshift(v##sign##int##bits##x##size vec1, vuint##bits##x##size vec2) \
	{ \
		VEC_AVX512F_RSHIFT_##bits##x##size(sign, l); \
	} \
	\
	static v##sign##int##bits##x##size##_impl v##sign##int##bits##x##size##_impl_avx512f = { \
		.load_aligned  = v##sign##int##bits##x##size##_avx512f_load_aligned, \
		.load          = v##sign##int##bits##x##size##_avx512f_load, \
		.store_aligned = v##sign##int##bits##x##size##_avx512f_store_aligned, \
		.store         = v##sign##int##bits##x##size##_avx512f_store, \
		.add           = v##sign##int##bits##x##size##_avx512f_add, \
		.sub           = v##sign##int##bits##x##size##_avx512f_sub, \
		.mul           = v##sign##int##bits##x##size##_avx512f_mul, \
		.and           = v##sign##int##bits##x##size##_avx512f_and, \
		.or            = v##sign##int##bits##x##size##_avx512f_or, \
		.xor           = v##sign##int##bits##x##size##_avx512f_xor, \
	};

#define VEC_AVX512F_DEFINE_OPERATIONS(bits, size) \
	VEC_AVX512F_DEFINE_OPERATIONS_SIGN( , bits, size) \
	VEC_AVX512F_DEFINE_OPERATIONS_SIGN(u, bits, size)

VEC_AVX512F_DEFINE_OPERATIONS(8, 64)
VEC_AVX512F_DEFINE_OPERATIONS(16, 32)
VEC_AVX512F_DEFINE_OPERATIONS(32, 16)
VEC_AVX512F_DEFINE_OPERATIONS(64, 8)

#undef VEC_AVX512F_DEFINE_OPERATIONS
#undef VEC_AVX512F_DEFINE_OPERATIONS_SIGN
#undef VEC_AVX512F_MUL_8x64
#undef VEC_AVX512F_MUL_16x32
#undef VEC_AVX512F_MUL_32x16
#undef VEC_AVX512F_MUL_64x8

#undef VEC_AVX512F_LSHIFT_8x64
#undef VEC_AVX512F_LSHIFT_16x32
#undef VEC_AVX512F_LSHIFT_32x16
#undef VEC_AVX512F_LSHIFT_64x8

#undef VEC_AVX512F_RSHIFT_8x64
#undef VEC_AVX512F_RSHIFT_16x32
#undef VEC_AVX512F_RSHIFT_32x16
#undef VEC_AVX512F_RSHIFT_64x8

#endif /* VEC_IMPL_X86_AVX512F_H_ */
