/**
 * vec - a tiny SIMD vector library in C99
 * 
 * Copyright (c) 2024 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

#include "vec/impl/x86/avx512f.h"

#include <immintrin.h>

/* ------------------------------------------------------------------------ */

#define VEC_AVX512F_MINMAX_TEMPLATE(SIGN, BITS, SIZE, INTLSIGN, OP) \
	VEC_FUNC_IMPL v##SIGN##int##BITS##x##SIZE v##SIGN##int##BITS##x##SIZE##_avx512f_##OP(v##SIGN##int##BITS##x##SIZE vec1, v##SIGN##int##BITS##x##SIZE vec2) \
	{ \
		union v##SIGN##int##BITS##x##SIZE##_impl_data *vec1d = (union v##SIGN##int##BITS##x##SIZE##_impl_data *)&vec1; \
		union v##SIGN##int##BITS##x##SIZE##_impl_data *vec2d = (union v##SIGN##int##BITS##x##SIZE##_impl_data *)&vec2; \
	\
		vec1d->avx512f = _mm512_##OP##_ep##INTLSIGN##BITS(vec1d->avx512f, vec2d->avx512f); \
	\
		return vec1d->vec; \
	}

#define VEC_AVX512F_MINMAX_32x16(OP)  VEC_AVX512F_MINMAX_TEMPLATE( , 32, 16, i, OP)
#define VEC_AVX512F_MINMAX_u32x16(OP) VEC_AVX512F_MINMAX_TEMPLATE(u, 32, 16, u, OP)
#define VEC_AVX512F_MINMAX_64x8(OP)   VEC_AVX512F_MINMAX_TEMPLATE( , 64, 8, i, OP)
#define VEC_AVX512F_MINMAX_u64x8(OP)  VEC_AVX512F_MINMAX_TEMPLATE(u, 64, 8, u, OP)

#define VEC_AVX512F_STRUCT_MINMAX_32x16(OP, SIGN)  v##SIGN##int32x16_avx512f_##OP
#define VEC_AVX512F_STRUCT_MINMAX_64x8(OP, SIGN)  v##SIGN##int64x8_avx512f_##OP

/* ------------------------------------------------------------------------ */

#define VEC_AVX512F_OPERATION_EX(name, op, sign, bits, size, secondsign) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_##name(v##sign##int##bits##x##size vec1, v##secondsign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##secondsign##int##bits##x##size##_impl_data *vec2d = (union v##secondsign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->avx512f = _mm512_##op##_epi##bits(vec1d->avx512f, vec2d->avx512f); \
	\
		return vec1d->vec; \
	}

#define VEC_AVX512F_OPERATION(name, op, sign, bits, size) \
	VEC_AVX512F_OPERATION_EX(name, op, sign, bits, size, sign)

#define VEC_AVX512F_OPERATION_SHIFT(name, op, sign, bits, size) \
	VEC_AVX512F_OPERATION_EX(name, op, sign, bits, size, u)

#define VEC_AVX512F_ADD_32x16(sign) VEC_AVX512F_OPERATION(add, add, sign, 32, 16)
#define VEC_AVX512F_ADD_64x8(sign) VEC_AVX512F_OPERATION(add, add, sign, 64, 8)

#define VEC_AVX512F_SUB_32x16(sign) VEC_AVX512F_OPERATION(sub, sub, sign, 32, 16)
#define VEC_AVX512F_SUB_64x8(sign) VEC_AVX512F_OPERATION(sub, sub, sign, 64, 8)

#define VEC_AVX512F_MUL_32x16(sign) VEC_AVX512F_OPERATION(mul, mullo, sign, 32, 16)
#define VEC_AVX512F_MUL_64x8(sign) VEC_AVX512F_OPERATION(mul, mullox, sign, 64, 8)

#define VEC_AVX512F_LSHIFT_32x16(sign) VEC_AVX512F_OPERATION_SHIFT(lshift, sllv, sign, 32, 16)
#define VEC_AVX512F_LSHIFT_64x8(sign)  VEC_AVX512F_OPERATION_SHIFT(lshift, sllv, sign, 64, 8)

#define VEC_AVX512F_XRSHIFT(name, bits, size, sign, aORl) \
	VEC_AVX512F_OPERATION_SHIFT(name, sr##aORl##v, sign, bits, size)

/* always the same, regardless of signedness */
#define VEC_AVX512F_LRSHIFT_32x16(sign) VEC_AVX512F_XRSHIFT(lrshift, 32, 16, sign, l)
#define VEC_AVX512F_LRSHIFT_64x8(sign) VEC_AVX512F_XRSHIFT(lrshift, 64, 8, sign, l)

#define VEC_AVX512F_RSHIFT_32x16(sign) VEC_AVX512F_XRSHIFT(rshift, 32, 16, sign, a)
#define VEC_AVX512F_RSHIFT_64x8(sign) VEC_AVX512F_XRSHIFT(rshift, 64, 8, sign, a)

#define VEC_AVX512F_uRSHIFT_32x16(sign) VEC_AVX512F_XRSHIFT(rshift, 32, 16, sign, l)
#define VEC_AVX512F_uRSHIFT_64x8(sign) VEC_AVX512F_XRSHIFT(rshift, 64, 8, sign, l)

/* ok */
#define VEC_AVX512F_STRUCT_OPERATION_32x16(OP, SIGN) v##SIGN##int32x16_avx512f_##OP
#define VEC_AVX512F_STRUCT_OPERATION_64x8(OP, SIGN)  v##SIGN##int64x8_avx512f_##OP

/* ------------------------------------------------------------------------ */

#define VEC_AVX512F_DEFINE_OPERATIONS_SIGN(sign, bits, size) \
	union v##sign##int##bits##x##size##_impl_data { \
		v##sign##int##bits##x##size vec; \
		__m512i avx512f; \
	}; \
	\
	VEC_STATIC_ASSERT(VEC_ALIGNOF(__m512i) <= VEC_ALIGNOF(v##sign##int##bits##x##size), "vec: v" #sign "int" #bits "x" #size " alignment needs to be expanded to fit intrinsic type size"); \
	VEC_STATIC_ASSERT(sizeof(__m512i) <= sizeof(v##sign##int##bits##x##size), "vec: v" #sign "int" #bits "x" #size " needs to be expanded to fit intrinsic type size"); \
	\
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_load_aligned(const vec_##sign##int##bits in[size]) \
	{ \
		union v##sign##int##bits##x##size##_impl_data vec; \
		vec.avx512f = _mm512_load_si512((const __m512i *)in); \
		return vec.vec; \
	} \
	\
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_load(const vec_##sign##int##bits in[size]) \
	{ \
		union v##sign##int##bits##x##size##_impl_data vec; \
		vec.avx512f = _mm512_loadu_si512((const __m512i *)in); \
		return vec.vec; \
	} \
	\
	VEC_FUNC_IMPL void v##sign##int##bits##x##size##_avx512f_store_aligned(v##sign##int##bits##x##size vec, vec_##sign##int##bits out[size]) \
	{ \
		_mm512_store_si512((__m512i *)out, ((union v##sign##int##bits##x##size##_impl_data *)&vec)->avx512f); \
	} \
	\
	VEC_FUNC_IMPL void v##sign##int##bits##x##size##_avx512f_store(v##sign##int##bits##x##size vec, vec_##sign##int##bits out[size]) \
	{ \
		_mm512_storeu_si512((__m512i *)out, ((union v##sign##int##bits##x##size##_impl_data *)&vec)->avx512f); \
	} \
	\
	VEC_AVX512F_ADD_##bits##x##size(sign) \
	VEC_AVX512F_SUB_##bits##x##size(sign) \
	VEC_AVX512F_MUL_##bits##x##size(sign) \
	\
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_and(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->avx512f = _mm512_and_si512(vec1d->avx512f, vec2d->avx512f); \
		return vec1d->vec; \
	} \
	\
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_or(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->avx512f = _mm512_or_si512(vec1d->avx512f, vec2d->avx512f); \
		return vec1d->vec; \
	} \
	\
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512f_xor(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->avx512f = _mm512_xor_si512(vec1d->avx512f, vec2d->avx512f); \
		return vec1d->vec; \
	} \
	\
	VEC_AVX512F_LSHIFT_##bits##x##size(sign); \
	VEC_AVX512F_##sign##RSHIFT_##bits##x##size(sign); \
	VEC_AVX512F_LRSHIFT_##bits##x##size(sign); \
	\
	VEC_AVX512F_MINMAX_##sign##bits##x##size(min) \
	VEC_AVX512F_MINMAX_##sign##bits##x##size(max) \
	\
	const v##sign##int##bits##x##size##_impl v##sign##int##bits##x##size##_impl_avx512f = { \
		.load_aligned = v##sign##int##bits##x##size##_avx512f_load_aligned, \
		.load = v##sign##int##bits##x##size##_avx512f_load, \
		.store_aligned = v##sign##int##bits##x##size##_avx512f_store_aligned, \
		.store = v##sign##int##bits##x##size##_avx512f_store, \
		.add = VEC_AVX512F_STRUCT_OPERATION_##bits##x##size(add, sign), \
		.sub = VEC_AVX512F_STRUCT_OPERATION_##bits##x##size(sub, sign), \
		.mul = VEC_AVX512F_STRUCT_OPERATION_##bits##x##size(mul, sign), \
		.band = v##sign##int##bits##x##size##_avx512f_and, \
		.bor = v##sign##int##bits##x##size##_avx512f_or, \
		.bxor = v##sign##int##bits##x##size##_avx512f_xor, \
		.lshift = VEC_AVX512F_STRUCT_OPERATION_##bits##x##size(lshift, sign), \
		.rshift = VEC_AVX512F_STRUCT_OPERATION_##bits##x##size(rshift, sign), \
		.lrshift = VEC_AVX512F_STRUCT_OPERATION_##bits##x##size(lrshift, sign), \
		.min = VEC_AVX512F_STRUCT_MINMAX_##bits##x##size(min, sign), \
		.max = VEC_AVX512F_STRUCT_MINMAX_##bits##x##size(max, sign), \
	};

#define VEC_AVX512F_DEFINE_OPERATIONS(bits, size) \
	VEC_AVX512F_DEFINE_OPERATIONS_SIGN(u, bits, size) \
	VEC_AVX512F_DEFINE_OPERATIONS_SIGN( , bits, size)

VEC_AVX512F_DEFINE_OPERATIONS(32, 16)
VEC_AVX512F_DEFINE_OPERATIONS(64, 8)
