/**
 * vec - a tiny SIMD vector library in C99
 * 
 * Copyright (c) 2024 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

#include "vec/impl/x86/sse42.h"
#include "vec/impl/generic.h"

#include <immintrin.h>

/* ------------------------------------------------------------------------ */
/* comparison */

#define MM_SET1_64(x) _mm_set1_epi64x(x)

/* helper funcs */
#define VEC_xSSE42_CMP(name, op, sign, bits, size, first, second, VARS, TRANS1, TRANS2) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_sse42_##name(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##sign##int##bits##x##size##_impl_data *vec2d = (union v##sign##int##bits##x##size##_impl_data *)&vec2; \
		VARS \
	\
		TRANS1 \
	\
		vec1d->sse = _mm_##op##_epi##bits(vec##first##d->sse, vec##second##d->sse); \
	\
		TRANS2 \
	\
		return vec1d->vec; \
	}

#define VEC_SSE42_CMP(name, op, bits, size, first, second) \
	VEC_xSSE42_CMP(name, op, /* nothing */, bits, size, first, second, /* nothing */, /* nothing */, /* nothing */)

#define VEC_uSSE42_CMP(name, op, bits, size, first, second) \
	VEC_xSSE42_CMP(name, op, u, bits, size, first, second, \
		__m128i xor_val = MM_SET1_##bits(UINT64_C(1) << (bits - 1)); \
	, { \
		vec1d->sse = _mm_xor_si128(vec1d->sse, xor_val); \
		vec2d->sse = _mm_xor_si128(vec2d->sse, xor_val); \
	}, \
	{ \
		/* nothing */ \
	})

/* these are the same for unsigned and signed, for obvious reasons. */
#define VEC_SSE42_CMPEQ_64x2(sign) VEC_xSSE42_CMP(cmpeq, cmpeq, sign, 64, 2, 1, 2, , ,)

/* ------------------------------------------------------------------------ */

#define VEC_SSE42_CMPLT_64x2(sign) VEC_##sign##SSE42_CMP(cmplt, cmpgt, 64, 2, 2, 1)
#define VEC_SSE42_CMPGT_64x2(sign) VEC_##sign##SSE42_CMP(cmpgt, cmpgt, 64, 2, 1, 2)

#define VEC_SSE42_STRUCT_CMP_64x2(name, sign) v##sign##int64x2_sse42_##name

/* ------------------------------------------------------------------------ */

// SSE 4.1 provides a real _mm_mullo_epi32
#define VEC_SSE42_DEFINE_OPERATIONS_SIGN(sign, bits, size) \
	union v##sign##int##bits##x##size##_impl_data { \
		v##sign##int##bits##x##size vec; \
		__m128i sse; \
	}; \
	\
	VEC_STATIC_ASSERT(VEC_ALIGNOF(__m128i) <= VEC_ALIGNOF(v##sign##int##bits##x##size), "vec: v" #sign "int32x4 alignment needs to be expanded to fit intrinsic type size"); \
	VEC_STATIC_ASSERT(sizeof(__m128i) <= sizeof(v##sign##int##bits##x##size), "vec: v" #sign "int32x4 needs to be expanded to fit intrinsic type size"); \
	\
	VEC_SSE42_CMPLT_##bits##x##size(sign); \
	VEC_SSE42_CMPGT_##bits##x##size(sign); \
	\
	const v##sign##int##bits##x##size##_impl v##sign##int##bits##x##size##_impl_sse42 = { \
		.cmplt = VEC_SSE42_STRUCT_CMP_##bits##x##size(cmplt, sign), \
		.cmpgt = VEC_SSE42_STRUCT_CMP_##bits##x##size(cmpgt, sign), \
	};

#define VEC_SSE42_DEFINE_OPERATIONS(bits, size) \
	VEC_SSE42_DEFINE_OPERATIONS_SIGN(u, bits, size) \
	VEC_SSE42_DEFINE_OPERATIONS_SIGN( , bits, size)

VEC_SSE42_DEFINE_OPERATIONS(64, 2)
