/**
 * vec - a tiny SIMD vector library in C99
 * 
 * Copyright (c) 2024 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

#include "vec/impl/x86/avx512bw.h"

#include <immintrin.h>

/* ------------------------------------------------------------------------ */

#define VEC_AVX512BW_MINMAX_TEMPLATE(SIGN, BITS, SIZE, INTLSIGN, OP) \
	VEC_FUNC_IMPL v##SIGN##int##BITS##x##SIZE v##SIGN##int##BITS##x##SIZE##_avx512bw_##OP(v##SIGN##int##BITS##x##SIZE vec1, v##SIGN##int##BITS##x##SIZE vec2) \
	{ \
		union v##SIGN##int##BITS##x##SIZE##_impl_data *vec1d = (union v##SIGN##int##BITS##x##SIZE##_impl_data *)&vec1; \
		union v##SIGN##int##BITS##x##SIZE##_impl_data *vec2d = (union v##SIGN##int##BITS##x##SIZE##_impl_data *)&vec2; \
	\
		vec1d->avx512bw = _mm512_##OP##_ep##INTLSIGN##BITS(vec1d->avx512bw, vec2d->avx512bw); \
	\
		return vec1d->vec; \
	}

#define VEC_AVX512BW_MINMAX_8x64(OP)   VEC_AVX512BW_MINMAX_TEMPLATE( , 8,  64, i, OP)
#define VEC_AVX512BW_MINMAX_u8x64(OP)  VEC_AVX512BW_MINMAX_TEMPLATE(u, 8,  64, u, OP)
#define VEC_AVX512BW_MINMAX_16x32(OP)  VEC_AVX512BW_MINMAX_TEMPLATE( , 16, 32, i, OP)
#define VEC_AVX512BW_MINMAX_u16x32(OP) VEC_AVX512BW_MINMAX_TEMPLATE(u, 16, 32, u, OP)

#define VEC_AVX512BW_STRUCT_MINMAX_8x64(OP, SIGN)  v##SIGN##int8x64_avx512bw_##OP
#define VEC_AVX512BW_STRUCT_MINMAX_16x32(OP, SIGN) v##SIGN##int16x32_avx512bw_##OP

/* ------------------------------------------------------------------------ */

#define VEC_AVX512BW_OPERATION_EX(name, op, sign, bits, size, secondsign) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_avx512bw_##name(v##sign##int##bits##x##size vec1, v##secondsign##int##bits##x##size vec2) \
	{ \
		union v##sign##int##bits##x##size##_impl_data *vec1d = (union v##sign##int##bits##x##size##_impl_data *)&vec1; \
		union v##secondsign##int##bits##x##size##_impl_data *vec2d = (union v##secondsign##int##bits##x##size##_impl_data *)&vec2; \
	\
		vec1d->avx512bw = _mm512_##op##_epi##bits(vec1d->avx512bw, vec2d->avx512bw); \
	\
		return vec1d->vec; \
	}

#define VEC_AVX512BW_OPERATION(name, op, sign, bits, size) \
	VEC_AVX512BW_OPERATION_EX(name, op, sign, bits, size, sign)

#define VEC_AVX512BW_OPERATION_SHIFT(name, op, sign, bits, size) \
	VEC_AVX512BW_OPERATION_EX(name, op, sign, bits, size, u)

#define VEC_AVX512BW_ADD_8x64(sign)  VEC_AVX512BW_OPERATION(add, add, sign, 8, 64)
#define VEC_AVX512BW_ADD_16x32(sign) VEC_AVX512BW_OPERATION(add, add, sign, 16, 32)

#define VEC_AVX512BW_SUB_8x64(sign)  VEC_AVX512BW_OPERATION(sub, sub, sign, 8, 64)
#define VEC_AVX512BW_SUB_16x32(sign) VEC_AVX512BW_OPERATION(sub, sub, sign, 16, 32)

#define VEC_AVX512BW_MUL_8x64(sign) /* nothing */
#define VEC_AVX512BW_MUL_16x32(sign) VEC_AVX512BW_OPERATION(mul, mullo, sign, 16, 32)

#define VEC_AVX512BW_LSHIFT_8x64(sign)  /* nothing */
#define VEC_AVX512BW_LSHIFT_16x32(sign) VEC_AVX512BW_OPERATION_SHIFT(lshift, sllv, sign, 16, 32)

#define VEC_AVX512BW_XRSHIFT(name, bits, size, sign, aORl) \
	VEC_AVX512BW_OPERATION_SHIFT(name, sr##aORl##v, sign, bits, size)

/* always the same, regardless of signedness */
#define VEC_AVX512BW_LRSHIFT_8x64(sign)  /* nothing */
#define VEC_AVX512BW_LRSHIFT_16x32(sign) VEC_AVX512BW_XRSHIFT(lrshift, 16, 32, sign, l)

#define VEC_AVX512BW_RSHIFT_8x64(sign)  /* nothing */
#define VEC_AVX512BW_RSHIFT_16x32(sign) VEC_AVX512BW_XRSHIFT(rshift, 16, 32, sign, a)

#define VEC_AVX512BW_uRSHIFT_8x64(sign)  /* nothing */
#define VEC_AVX512BW_uRSHIFT_16x32(sign) VEC_AVX512BW_XRSHIFT(rshift, 16, 32, sign, l)

/* ok */
#define VEC_AVX512BW_STRUCT_ADDSUB_8x64(OP, SIGN)  v##SIGN##int8x64_avx512bw_##OP
#define VEC_AVX512BW_STRUCT_ADDSUB_16x32(OP, SIGN) v##SIGN##int16x32_avx512bw_##OP

#define VEC_AVX512BW_STRUCT_OPERATION_8x64(OP, SIGN)  NULL
#define VEC_AVX512BW_STRUCT_OPERATION_16x32(OP, SIGN) v##SIGN##int16x32_avx512bw_##OP

/* ------------------------------------------------------------------------ */

#define VEC_AVX512BW_DEFINE_OPERATIONS_SIGN(sign, bits, size) \
	union v##sign##int##bits##x##size##_impl_data { \
		v##sign##int##bits##x##size vec; \
		__m512i avx512bw; \
	}; \
	\
	VEC_STATIC_ASSERT(VEC_ALIGNOF(__m512i) <= VEC_ALIGNOF(v##sign##int##bits##x##size), "vec: v" #sign "int" #bits "x" #size " alignment needs to be expanded to fit intrinsic type size"); \
	VEC_STATIC_ASSERT(sizeof(__m512i) <= sizeof(v##sign##int##bits##x##size), "vec: v" #sign "int" #bits "x" #size " needs to be expanded to fit intrinsic type size"); \
	\
	VEC_AVX512BW_ADD_##bits##x##size(sign) \
	VEC_AVX512BW_SUB_##bits##x##size(sign) \
	VEC_AVX512BW_MUL_##bits##x##size(sign) \
	\
	VEC_AVX512BW_LSHIFT_##bits##x##size(sign); \
	VEC_AVX512BW_##sign##RSHIFT_##bits##x##size(sign); \
	VEC_AVX512BW_LRSHIFT_##bits##x##size(sign); \
	\
	VEC_AVX512BW_MINMAX_##sign##bits##x##size(min) \
	VEC_AVX512BW_MINMAX_##sign##bits##x##size(max) \
	\
	const v##sign##int##bits##x##size##_impl v##sign##int##bits##x##size##_impl_avx512bw = { \
		.add = VEC_AVX512BW_STRUCT_ADDSUB_##bits##x##size(add, sign), \
		.sub = VEC_AVX512BW_STRUCT_ADDSUB_##bits##x##size(sub, sign), \
		.mul = VEC_AVX512BW_STRUCT_OPERATION_##bits##x##size(mul, sign), \
		.lshift = VEC_AVX512BW_STRUCT_OPERATION_##bits##x##size(lshift, sign), \
		.rshift = VEC_AVX512BW_STRUCT_OPERATION_##bits##x##size(rshift, sign), \
		.lrshift = VEC_AVX512BW_STRUCT_OPERATION_##bits##x##size(lrshift, sign), \
		.min = VEC_AVX512BW_STRUCT_MINMAX_##bits##x##size(min, sign), \
		.max = VEC_AVX512BW_STRUCT_MINMAX_##bits##x##size(max, sign), \
	};

#define VEC_AVX512BW_DEFINE_OPERATIONS(bits, size) \
	VEC_AVX512BW_DEFINE_OPERATIONS_SIGN(u, bits, size) \
	VEC_AVX512BW_DEFINE_OPERATIONS_SIGN( , bits, size)

VEC_AVX512BW_DEFINE_OPERATIONS(8, 64)
VEC_AVX512BW_DEFINE_OPERATIONS(16, 32)
