/**
 * vec - a tiny SIMD vector library in C99
 * 
 * Copyright (c) 2024-2025 Paper
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
**/

#ifndef VEC_IMPL_X86_SSE2_H_
#define VEC_IMPL_X86_SSE2_H_

#include <emmintrin.h>

/* eh */
#define VEC_SSE2_SET1_8(x)  _mm_set1_epi8(x)
#define VEC_SSE2_SET1_16(x) _mm_set1_epi16(x)
#define VEC_SSE2_SET1_32(x) _mm_set1_epi32(x)
#define VEC_SSE2_SET1_64(x) _mm_set1_epi64x(x)

/* ------------------------------------------------------------------------ */

/* despite this macro's name, it's used to basically define every single
 * operation :) (with a few exceptions) */
#define VEC_SSE2_OP_EX(name, op, sign, bits, size, first, second, VARS, TRANS1, TRANS2, INTLSIGN) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_##name(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		VARS \
	\
		TRANS1 \
	\
		vec1.sse = _mm_##op##_ep##INTLSIGN##bits(vec##first.sse, vec##second.sse); \
	\
		TRANS2 \
	\
		return vec1; \
	}

#define VEC_SSE2_OP_XOR(name, op, sign, bits, size, first, second, intlsign) \
	VEC_SSE2_OP_EX(name, op, sign, bits, size, first, second, \
		__m128i xor_val = VEC_SSE2_SET1_##bits((vec_int##bits)(1u << (bits - 1))); \
	, { \
		vec1.sse = _mm_xor_si128(vec1.sse, xor_val); \
		vec2.sse = _mm_xor_si128(vec2.sse, xor_val); \
	}, \
	{ \
		vec1.sse = _mm_xor_si128(vec1.sse, xor_val); \
	}, intlsign)

#define VEC_SSE2_OP(name, op, sign, bits, size, first, second, intlsign) \
	VEC_SSE2_OP_EX(name, op, sign, bits, size, first, second, /* nothing */, /* nothing */, /* nothing */, intlsign)

/* ------------------------------------------------------------------------ */
/* comparison */

#define VEC_xSSE2_CMP(name, op, sign, bits, size, first, second, VARS, TRANS1, TRANS2) \
	VEC_SSE2_OP_EX(name, op, sign, bits, size, first, second, VARS, TRANS1, TRANS2, i)

#define VEC_SSE2_CMP(name, op, bits, size, first, second) \
	VEC_xSSE2_CMP(name, op, /* nothing */, bits, size, first, second, /* nothing */, /* nothing */, /* nothing */)

#define VEC_uSSE2_CMP(name, op, bits, size, first, second) \
	VEC_xSSE2_CMP(name, op, u, bits, size, first, second, \
		__m128i xor_val = VEC_SSE2_SET1_##bits((vec_int##bits)(1u << (bits - 1))); \
	, { \
		vec1.sse = _mm_xor_si128(vec1.sse, xor_val); \
		vec2.sse = _mm_xor_si128(vec2.sse, xor_val); \
	}, \
	{ \
		/* nothing */ \
	})

/* these are the same for unsigned and signed, for obvious reasons. */
#define VEC_SSE2_CMPEQ_8x16(sign) VEC_xSSE2_CMP(cmpeq, cmpeq, sign, 8, 16, 1, 2, , ,)
#define VEC_SSE2_CMPEQ_16x8(sign) VEC_xSSE2_CMP(cmpeq, cmpeq, sign, 16, 8, 1, 2, , ,)
#define VEC_SSE2_CMPEQ_32x4(sign) VEC_xSSE2_CMP(cmpeq, cmpeq, sign, 32, 4, 1, 2, , ,)

/* SSE2 doesn't have an intrinsic for 64x2 equality comparison,
 * so how can we take a 32x4 comparison result and turn it into
 * a 64x2 comparison result?
 *
 * well, Intel conveniently provided an operation where we can
 * shuffle around 32-bit integers (_mm_shuffle_epi32).
 *
 * this means all we have to do is simply do the 32-bit operation,
 * shuffle the parts, and then return a bitwise AND of the result. */

#define VEC_SSE2_CMPEQ_64x2(sign) \
	VEC_FUNC_IMPL v##sign##int64x2 v##sign##int64x2_cmpeq(v##sign##int64x2 vec1, v##sign##int64x2 vec2) \
	{ \
		vec1.sse = _mm_cmpeq_epi32(vec1.sse, vec2.sse); \
		vec2.sse = _mm_shuffle_epi32(vec1.sse, _MM_SHUFFLE(1, 1, 3, 3)); \
		vec1.sse = _mm_shuffle_epi32(vec1.sse, _MM_SHUFFLE(0, 0, 2, 2)); \
		vec1.sse = _mm_and_si128(vec1.sse, vec2.sse); \
	\
		return vec1; \
	}

/* ------------------------------------------------------------------------ */

#define VEC_SSE2_SPLAT(sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_splat(vec_##sign##int##bits x) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.sse = VEC_SSE2_SET1_##bits(x); \
		return vec; \
	}

#define VEC_SSE2_LOAD_ALIGNED(sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_load_aligned(const vec_##sign##int##bits in[size]) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.sse = _mm_load_si128((const __m128i *)in); \
		return vec; \
	}

#define VEC_SSE2_LOAD(sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_load(const vec_##sign##int##bits in[size]) \
	{ \
		v##sign##int##bits##x##size vec; \
		vec.sse = _mm_loadu_si128((const __m128i *)in); \
		return vec; \
	}

#define VEC_SSE2_STORE_ALIGNED(sign, bits, size) \
	VEC_FUNC_IMPL void v##sign##int##bits##x##size##_store_aligned(v##sign##int##bits##x##size vec, vec_##sign##int##bits out[size]) \
	{ \
		_mm_store_si128((__m128i *)out, vec.sse); \
	}

#define VEC_SSE2_STORE(sign, bits, size) \
	VEC_FUNC_IMPL void v##sign##int##bits##x##size##_store(v##sign##int##bits##x##size vec, vec_##sign##int##bits out[size]) \
	{ \
		_mm_storeu_si128((__m128i *)out, vec.sse); \
	}

#define VEC_SSE2_ADD(sign, bits, size) \
	VEC_SSE2_OP(add, add, sign, bits, size, 1, 2, i)

#define VEC_SSE2_SUB(sign, bits, size) \
	VEC_SSE2_OP(sub, sub, sign, bits, size, 1, 2, i)

#define VEC_SSE2_AND(sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_and(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		vec1.sse = _mm_and_si128(vec1.sse, vec2.sse); \
		return vec1; \
	}

#define VEC_SSE2_OR(sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_or(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		vec1.sse = _mm_or_si128(vec1.sse, vec2.sse); \
		return vec1; \
	}

#define VEC_SSE2_XOR(sign, bits, size) \
	VEC_FUNC_IMPL v##sign##int##bits##x##size v##sign##int##bits##x##size##_xor(v##sign##int##bits##x##size vec1, v##sign##int##bits##x##size vec2) \
	{ \
		vec1.sse = _mm_xor_si128(vec1.sse, vec2.sse); \
		return vec1; \
	}

/* ------------------------------------------------------------------------ */
/* vint8x16 */

#ifndef VINT8x16_SPLAT_DEFINED
VEC_SSE2_SPLAT(/* nothing */, 8, 16)
# define VINT8x16_SPLAT_DEFINED
#endif

#ifndef VINT8x16_LOAD_ALIGNED_DEFINED
VEC_SSE2_LOAD_ALIGNED(/* nothing */, 8, 16)
# define VINT8x16_LOAD_ALIGNED_DEFINED
#endif

#ifndef VINT8x16_LOAD_DEFINED
VEC_SSE2_LOAD(/* nothing */, 8, 16)
# define VINT8x16_LOAD_DEFINED
#endif

#ifndef VINT8x16_STORE_ALIGNED_DEFINED
VEC_SSE2_STORE_ALIGNED(/* nothing */, 8, 16)
# define VINT8x16_STORE_ALIGNED_DEFINED
#endif

#ifndef VINT8x16_STORE_DEFINED
VEC_SSE2_STORE(/* nothing */, 8, 16)
# define VINT8x16_STORE_DEFINED
#endif

#ifndef VINT8x16_ADD_DEFINED
VEC_SSE2_ADD(/* nothing */, 8, 16)
# define VINT8x16_ADD_DEFINED
#endif

#ifndef VINT8x16_SUB_DEFINED
VEC_SSE2_SUB(/* nothing */, 8, 16)
# define VINT8x16_SUB_DEFINED
#endif

#ifndef VINT8x16_AND_DEFINED
VEC_SSE2_AND(/* nothing */, 8, 16)
# define VINT8x16_AND_DEFINED
#endif

#ifndef VINT8x16_OR_DEFINED
VEC_SSE2_OR(/* nothing */, 8, 16)
# define VINT8x16_OR_DEFINED
#endif

#ifndef VINT8x16_XOR_DEFINED
VEC_SSE2_XOR(/* nothing */, 8, 16)
# define VINT8x16_XOR_DEFINED
#endif

#ifndef VINT8x16_CMPGT_DEFINED
VEC_SSE2_CMP(cmpgt, cmpgt, 8, 16, 1, 2)
# define VINT8x16_CMPGT_DEFINED
#endif

#ifndef VINT8x16_CMPLT_DEFINED
VEC_SSE2_CMP(cmplt, cmpgt, 8, 16, 2, 1)
# define VINT8x16_CMPLT_DEFINED
#endif

#ifndef VINT8x16_CMPEQ_DEFINED
VEC_xSSE2_CMP(cmpeq, cmpeq, /* nothing */, 8, 16, 1, 2, , ,)
# define VINT8x16_CMPEQ_DEFINED
#endif

#ifndef VINT8x16_MIN_DEFINED
VEC_SSE2_OP_XOR(min, min, /* nothing */, 8, 16, 1, 2, u)
# define VINT8x16_MIN_DEFINED
#endif

#ifndef VINT8x16_MAX_DEFINED
VEC_SSE2_OP_XOR(max, max, /* nothing */, 8, 16, 1, 2, u)
# define VINT8x16_MAX_DEFINED
#endif

/* ------------------------------------------------------------------------ */
/* vuint8x16 */

#ifndef VUINT8x16_SPLAT_DEFINED
VEC_SSE2_SPLAT(u, 8, 16)
# define VUINT8x16_SPLAT_DEFINED
#endif

#ifndef VUINT8x16_LOAD_ALIGNED_DEFINED
VEC_SSE2_LOAD_ALIGNED(u, 8, 16)
# define VUINT8x16_LOAD_ALIGNED_DEFINED
#endif

#ifndef VUINT8x16_LOAD_DEFINED
VEC_SSE2_LOAD(u, 8, 16)
# define VUINT8x16_LOAD_DEFINED
#endif

#ifndef VUINT8x16_STORE_ALIGNED_DEFINED
VEC_SSE2_STORE_ALIGNED(u, 8, 16)
# define VUINT8x16_STORE_ALIGNED_DEFINED
#endif

#ifndef VUINT8x16_STORE_DEFINED
VEC_SSE2_STORE(u, 8, 16)
# define VUINT8x16_STORE_DEFINED
#endif

#ifndef VUINT8x16_ADD_DEFINED
VEC_SSE2_ADD(u, 8, 16)
# define VUINT8x16_ADD_DEFINED
#endif

#ifndef VUINT8x16_SUB_DEFINED
VEC_SSE2_SUB(u, 8, 16)
# define VUINT8x16_SUB_DEFINED
#endif

#ifndef VUINT8x16_AND_DEFINED
VEC_SSE2_AND(u, 8, 16)
# define VUINT8x16_AND_DEFINED
#endif

#ifndef VUINT8x16_OR_DEFINED
VEC_SSE2_OR(u, 8, 16)
# define VUINT8x16_OR_DEFINED
#endif

#ifndef VUINT8x16_XOR_DEFINED
VEC_SSE2_XOR(u, 8, 16)
# define VUINT8x16_XOR_DEFINED
#endif

#ifndef VUINT8x16_CMPGT_DEFINED
VEC_uSSE2_CMP(cmpgt, cmpgt, 8, 16, 1, 2)
# define VUINT8x16_CMPGT_DEFINED
#endif

#ifndef VUINT8x16_CMPLT_DEFINED
VEC_uSSE2_CMP(cmplt, cmpgt, 8, 16, 2, 1)
# define VUINT8x16_CMPLT_DEFINED
#endif

#ifndef VUINT8x16_CMPEQ_DEFINED
VEC_xSSE2_CMP(cmpeq, cmpeq, u, 8, 16, 1, 2, , ,)
# define VUINT8x16_CMPEQ_DEFINED
#endif

#ifndef VUINT8x16_MIN_DEFINED
VEC_SSE2_OP(min, min, u, 8, 16, 1, 2, u)
# define VUINT8x16_MIN_DEFINED
#endif

#ifndef VUINT8x16_MAX_DEFINED
VEC_SSE2_OP(max, max, u, 8, 16, 1, 2, u)
# define VUINT8x16_MAX_DEFINED
#endif

/* ------------------------------------------------------------------------ */
/* vint8x16 */

#ifndef VINT16x8_SPLAT_DEFINED
VEC_SSE2_SPLAT(/* nothing */, 16, 8)
# define VINT16x8_SPLAT_DEFINED
#endif

#ifndef VINT16x8_LOAD_ALIGNED_DEFINED
VEC_SSE2_LOAD_ALIGNED(/* nothing */, 16, 8)
# define VINT16x8_LOAD_ALIGNED_DEFINED
#endif

#ifndef VINT16x8_LOAD_DEFINED
VEC_SSE2_LOAD(/* nothing */, 16, 8)
# define VINT16x8_LOAD_DEFINED
#endif

#ifndef VINT16x8_STORE_ALIGNED_DEFINED
VEC_SSE2_STORE_ALIGNED(/* nothing */, 16, 8)
# define VINT16x8_STORE_ALIGNED_DEFINED
#endif

#ifndef VINT16x8_STORE_DEFINED
VEC_SSE2_STORE(/* nothing */, 16, 8)
# define VINT16x8_STORE_DEFINED
#endif

#ifndef VINT16x8_ADD_DEFINED
VEC_SSE2_ADD(/* nothing */, 16, 8)
# define VINT16x8_ADD_DEFINED
#endif

#ifndef VINT16x8_SUB_DEFINED
VEC_SSE2_SUB(/* nothing */, 16, 8)
# define VINT16x8_SUB_DEFINED
#endif

#ifndef VINT16x8_MUL_DEFINED
VEC_SSE2_OP(mul, mullo, /* nothing */, 16, 8, 1, 2, i)
# define VINT16x8_MUL_DEFINED
#endif

#ifndef VINT16x8_AND_DEFINED
VEC_SSE2_AND(/* nothing */, 16, 8)
# define VINT16x8_AND_DEFINED
#endif

#ifndef VINT16x8_OR_DEFINED
VEC_SSE2_OR(/* nothing */, 16, 8)
# define VINT16x8_OR_DEFINED
#endif

#ifndef VINT16x8_XOR_DEFINED
VEC_SSE2_XOR(/* nothing */, 16, 8)
# define VINT16x8_XOR_DEFINED
#endif

#ifndef VINT16x8_CMPGT_DEFINED
VEC_SSE2_CMP(cmpgt, cmpgt, 16, 8, 1, 2)
# define VINT16x8_CMPGT_DEFINED
#endif

#ifndef VINT16x8_CMPLT_DEFINED
VEC_SSE2_CMP(cmplt, cmpgt, 16, 8, 2, 1)
# define VINT16x8_CMPLT_DEFINED
#endif

#ifndef VINT16x8_CMPEQ_DEFINED
VEC_xSSE2_CMP(cmpeq, cmpeq, /* nothing */, 16, 8, 1, 2, , ,)
# define VINT16x8_CMPEQ_DEFINED
#endif

#ifndef VINT16x8_MIN_DEFINED
VEC_SSE2_OP(min, min, /* nothing */, 16, 8, 1, 2, i)
# define VINT16x8_MIN_DEFINED
#endif

#ifndef VINT16x8_MAX_DEFINED
VEC_SSE2_OP(max, max, /* nothing */, 16, 8, 1, 2, i)
# define VINT16x8_MAX_DEFINED
#endif

/* ------------------------------------------------------------------------ */
/* vuint8x16 */

#ifndef VUINT16x8_SPLAT_DEFINED
VEC_SSE2_SPLAT(u, 16, 8)
# define VUINT16x8_SPLAT_DEFINED
#endif

#ifndef VUINT16x8_LOAD_ALIGNED_DEFINED
VEC_SSE2_LOAD_ALIGNED(u, 16, 8)
# define VUINT16x8_LOAD_ALIGNED_DEFINED
#endif

#ifndef VUINT16x8_LOAD_DEFINED
VEC_SSE2_LOAD(u, 16, 8)
# define VUINT16x8_LOAD_DEFINED
#endif

#ifndef VUINT16x8_STORE_ALIGNED_DEFINED
VEC_SSE2_STORE_ALIGNED(u, 16, 8)
# define VUINT16x8_STORE_ALIGNED_DEFINED
#endif

#ifndef VUINT16x8_STORE_DEFINED
VEC_SSE2_STORE(u, 16, 8)
# define VUINT16x8_STORE_DEFINED
#endif

#ifndef VUINT16x8_ADD_DEFINED
VEC_SSE2_ADD(u, 16, 8)
# define VUINT16x8_ADD_DEFINED
#endif

#ifndef VUINT16x8_SUB_DEFINED
VEC_SSE2_SUB(u, 16, 8)
# define VUINT16x8_SUB_DEFINED
#endif

#ifndef VUINT16x8_MUL_DEFINED
VEC_SSE2_OP(mul, mullo, u, 16, 8, 1, 2, i)
# define VUINT16x8_MUL_DEFINED
#endif

#ifndef VUINT16x8_AND_DEFINED
VEC_SSE2_AND(u, 16, 8)
# define VUINT16x8_AND_DEFINED
#endif

#ifndef VUINT16x8_OR_DEFINED
VEC_SSE2_OR(u, 16, 8)
# define VUINT16x8_OR_DEFINED
#endif

#ifndef VUINT16x8_XOR_DEFINED
VEC_SSE2_XOR(u, 16, 8)
# define VUINT16x8_XOR_DEFINED
#endif

#ifndef VUINT16x8_CMPGT_DEFINED
VEC_uSSE2_CMP(cmpgt, cmpgt, 16, 8, 1, 2)
# define VUINT16x8_CMPGT_DEFINED
#endif

#ifndef VUINT16x8_CMPLT_DEFINED
VEC_uSSE2_CMP(cmplt, cmpgt, 16, 8, 2, 1)
# define VUINT16x8_CMPLT_DEFINED
#endif

#ifndef VUINT16x8_CMPEQ_DEFINED
VEC_xSSE2_CMP(cmpeq, cmpeq, u, 16, 8, 1, 2, , ,)
# define VUINT16x8_CMPEQ_DEFINED
#endif

#ifndef VUINT16x8_MIN_DEFINED
VEC_SSE2_OP_XOR(min, min, u, 16, 8, 1, 2, i)
# define VUINT16x8_MIN_DEFINED
#endif

#ifndef VUINT16x8_MAX_DEFINED
VEC_SSE2_OP_XOR(max, max, u, 16, 8, 1, 2, i)
# define VUINT16x8_MAX_DEFINED
#endif

/* ------------------------------------------------------------------------ */
/* vint64x2 */

/* many things are more difficult with 64-bit values */
#define VEC_SSE2_CMPEQ_64x2(sign) \
	VEC_FUNC_IMPL v##sign##int64x2 v##sign##int64x2_cmpeq(v##sign##int64x2 vec1, v##sign##int64x2 vec2) \
	{ \
		vec1.sse = _mm_cmpeq_epi32(vec1.sse, vec2.sse); \
		vec2.sse = _mm_shuffle_epi32(vec1.sse, _MM_SHUFFLE(1, 1, 3, 3)); \
		vec1.sse = _mm_shuffle_epi32(vec1.sse, _MM_SHUFFLE(0, 0, 2, 2)); \
		vec1.sse = _mm_and_si128(vec1.sse, vec2.sse); \
	\
		return vec1; \
	}

#define VEC_SSE2_MUL_64x2(sign) \
	VEC_FUNC_IMPL v##sign##int64x2 v##sign##int64x2_mul(v##sign##int64x2 vec1, v##sign##int64x2 vec2) \
	{ \
		__m128i ac = _mm_mul_epu32(vec1.sse, vec2.sse); /* ac = (vec1 & UINT32_MAX) * (vec2 & UINT32_MAX); */ \
		__m128i b  = _mm_srli_epi64(vec1.sse, 32);      /* b = vec1 >> 32; */ \
		__m128i bc = _mm_mul_epu32(b, vec2.sse);        /* bc = b * (vec2 & UINT32_MAX); */ \
		__m128i d  = _mm_srli_epi64(vec2.sse, 32);      /* d = vec2 >> 32; */ \
		__m128i ad = _mm_mul_epu32(vec1.sse, d);        /* ad = (vec1 & UINT32_MAX) * d; */ \
		__m128i hi = _mm_add_epi64(bc, ad);             /* hi = bc + ad; */ \
		hi = _mm_slli_epi64(hi, 32);                    /* hi <<= 32; */ \
	\
		vec1.sse = _mm_add_epi64(hi, ac); /* (ab3,ab2,ab1,ab0) */ \
		return vec1; \
	}

#ifndef VINT64x2_MUL_DEFINED
VEC_SSE2_MUL_64x2(/* nothing */)
# define VINT64x2_MUL_DEFINED
#endif

#ifndef VINT64x2_CMPEQ_DEFINED
VEC_SSE2_CMPEQ_64x2(/* nothing */)
# define VINT64x2_CMPEQ_DEFINED
#endif

/* ------------------------------------------------------------------------ */
/* vuint64x2 */

#ifndef VUINT64x2_MUL_DEFINED
VEC_SSE2_MUL_64x2(u)
# define VUINT64x2_MUL_DEFINED
#endif

#ifndef VUINT64x2_CMPEQ_DEFINED
VEC_SSE2_CMPEQ_64x2(u)
# define VUINT64x2_CMPEQ_DEFINED
#endif

#endif /* VEC_IMPL_X86_SSE2_H_ */
